%
% [CLASSES] = SPANALYZE(QUERIES, DATABASE, ...)
%
% Attribute points by eigen analysis.
%
%   Copyright 2012 Kristian L. Damkjer
%
%   Software History:
%      2013-JAN-28   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Change output to a struct for easier referencing.
%

function [ CLASSES ] = spanalyze( QUERIES, DATABASE, varargin )
% Parse Inputs
userParams = parseInputs(varargin{:});

% disp(userParams);

dimensions=size(QUERIES, 1);
elements=size(QUERIES, 2);

norms=zeros(size(QUERIES));
feats=zeros(size(QUERIES));

biases=zeros(1,elements);

ints=zeros(1,elements);

msg='Computing Structure Tensors...';
tstart=tic;
h = timebar(1, elements, msg, tstart);

step=1000;
tic;

% maxnbrs = 0;

for elem=1:step:elements
   
   last=min(elem+step-1,elements);
   
   % Get the nearest neighbors of elem
   if (userParams.radius > 0 && userParams.neighbors <= 0)
      % Perform a fixed radius search
      NN=DATABASE.rnn(QUERIES(:,elem:last),...
                      userParams.radius);
   elseif (userParams.radius <= 0 && userParams.neighbors > 0)
      % Search unconstrained neighbors
      NN=DATABASE.knn(QUERIES(:,elem:last),...
                      userParams.neighbors);
   elseif (userParams.radius > 0 && userParams.neighbors > 0)
      % Search constrained to radius
      NN=DATABASE.knn(QUERIES(:,elem:last),...
                      userParams.neighbors,...
                      'lim',userParams.radius);
   elseif (~isempty(userParams.counts) && userParams.radius <= 0)
      % Search unconstrained neighbors
      NN=DATABASE.knn(QUERIES(:,elem:last),...
                      max(userParams.counts));
   elseif (~isempty(userParams.counts) && userParams.radius > 0)
      % Search constrained to radius
      NN=DATABASE.knn(QUERIES(:,elem:last),...
                      max(userParams.counts),...
                      'lim',userParams.radius);
   elseif (~isempty(userParams.steps))
      % Perform a fixed radius search
      [NN,DISTS]=DATABASE.rnn(QUERIES(:,elem:last),...
                              max(userParams.steps));
   end

   if (isempty(userParams.counts) && isempty(userParams.steps))
      cells=cell(1,length(NN));
      
      for n=1:length(NN)
         cells{n}=QUERIES(:,NN{n})';
      end
 
%      covs=cell(1,length(cells));
%      
%      parfor n=1:length(cells)
%         covs{n}=cov(cells{n}');
%      end
      
%      [covs,bias,inty]=fastcov(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));
      [covs,bias,inty]=fastcov(cells);
   elseif (~isempty(userParams.counts))
      tempfeats=zeros(dimensions,size(NN,1));
      tempde=zeros(length(userParams.counts),size(NN,1));

      tempcovs=cell(length(userParams.counts),1);
      tempbias=cell(length(userParams.counts),1);
      tempinty=cell(length(userParams.counts),1);
      
      for c=1:length(userParams.counts)
         [tempcovs{c},tempbias{c},tempinty{c}]=fastcov(cellfun(@(x) QUERIES(:,x(1:userParams.counts(c)))',NN,'UniformOutput',false));

         for nbr=1:size(NN,1)
            [~,D]=eig(tempcovs{c}{nbr});
            tempfeats(:,nbr)=sort(diag(D),'descend');
         end
         
         % Compute dimensional degree
         tempdims=(tempfeats(1:dimensions-1,:)-tempfeats(2:dimensions,:))./...
                   repmat(tempfeats(1,:),dimensions-1,1);

         % Compute isotropy
         tempiso=(tempfeats(dimensions,:))./(tempfeats(1,:));
         
         % Compute dimensional entropy
         tempalpha=[tempdims;tempiso];
         tempde(c,:)=-sum(tempalpha.*log(tempalpha))./log(dimensions);
      end
      
      [~,ind]=min(tempde);

      covs=tempcovs{ind};
      bias=tempbias{ind};
      inty=tempinty{ind};
   elseif (~isempty(userParams.steps))
      tempfeats=zeros(dimensions,size(NN,1));
      tempde=zeros(length(userParams.steps),size(NN,1));

      tempcovs=cell(length(userParams.steps),1);
      tempbias=cell(length(userParams.steps),1);
      tempinty=cell(length(userParams.steps),1);
      
      for c=1:length(userParams.steps)
%          disp('userParams.steps(c)');
%          disp(userParams.steps(c));

%          disp('NN');
%          disp(NN);
         
%          disp('DISTS');
%          disp(DISTS);
         
%          disp('DISTS <= userParams.steps(c)');
%          cellfun(@(x) disp(x <= userParams.steps(c))',DISTS,'UniformOutput',false);

         nbrs=cellfun(@(x) sum(x <= userParams.steps(c))',DISTS,'UniformOutput',false);
         
%          disp('NN(DISTS <= userParams.steps(c))');
%          cellfun(@(x,y) disp(x(y <= userParams.steps(c))'),NN,DISTS,'UniformOutput',false)
         
%          disp('QUERIES(:,NN(DISTS <= userParams.steps(c)))');
%          cellfun(@(x,y) disp(QUERIES(:,x(y <= userParams.steps(c)))'),NN,DISTS,'UniformOutput',false)

         [tempcovs{c},tempbias{c},tempinty{c}]=fastcov(cellfun(@(x,y) QUERIES(:,x(y <= userParams.steps(c)))',NN,DISTS,'UniformOutput',false));

%          disp('tempcovs{c}');
%          cellfun(@(x) disp(x), tempcovs{c});
         
         for nbr=1:size(NN,1)
            if (nbrs{nbr}<5)
               continue;
            end

            [~,D]=eig(tempcovs{c}{nbr});
            tempfeats(:,nbr)=sort(diag(D),'descend');
         end

%          disp('tempfeats');
%          disp(tempfeats);

         % Compute dimensional degree
         % Do not use the following definition based on singular values:
         % tempdims=(sqrt(tempfeats(1:dimensions-1,:))-sqrt(tempfeats(2:dimensions,:)))./...
         %           repmat(sqrt(tempfeats(1,:)),dimensions-1,1);

         tempdims=(tempfeats(1:dimensions-1,:)-tempfeats(2:dimensions,:))./...
                   repmat(tempfeats(1,:),dimensions-1,1);

         % Compute isotropy
         tempiso=(tempfeats(dimensions,:))./(tempfeats(1,:));
         
         % Compute dimensional entropy
         tempalpha=[tempdims;tempiso];
         tempde(c,:)=-sum(tempalpha.*log(tempalpha))./log(dimensions);
      end

%       disp('tempde');
%       disp(tempde);

      [~,ind]=nanmin(tempde);

      covs=tempcovs{ind};
      bias=tempbias{ind};
      inty=tempinty{ind};

   else
      error('What?!');
   end

% Same solution via SVD appears to be slower based on emperical testing
%    recentered=fastcenter(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));

%    disp('covs');
%    cellfun(@(x) disp(x),covs);

   rad=userParams.radius;

   [V,D]=par_eig(covs);
   
   for nbr=1:size(NN,1)
      % skip underconstrained (possible with radius searches)
      if (rad > 0 && (length(NN{nbr})<5 || any(any(isnan(covs{nbr})))))
         continue;
      end

%       maxnbrs=max(maxnbrs,length(NN{nbr}));
      
%      [V,D]=eig(covs{nbr});
%      [~,index]=min(diag(D));
%      [~,index]=min(D{nbr});
%       norms(:,elem+nbr-1)=abs(V(:,index)) .* sqrt(D(index, index));
%      norms(:,elem+nbr-1)=abs(V(:,index));
%      norms(:,elem+nbr-1)=abs(V{nbr}(:,index));

      A=V{nbr};
      norms(:,elem+nbr-1)=abs(A(:,end));
      
      % Define features as eigenvalues
      % feats(:,elem+nbr-1)=sort(diag(D),'descend');

      % Define features as singular values
%      feats(:,elem+nbr-1)=sqrt(sort(diag(D),'descend'));
%      feats(:,elem+nbr-1)=sqrt(sort(D{nbr},'descend'));
      feats(:,elem+nbr-1)=sqrt(D{nbr});

      biases(elem+nbr-1)=bias{nbr};
      ints(elem+nbr-1)=inty{nbr};

% Same solution via SVD appears to be slower via emperical testing
%       [~,S,V]=svd(recentered{nbr},0);
%       S = S ./ sqrt(length(NN{nbr})-1);
%       norms(:,elem+nbr-1)=abs(V(:,dimensions) .* S(dimensions, dimensions));
%       feats(:,elem+nbr-1)=diag(S).^2;
      
   end
   
   if (toc > 1)
      tic;
      h = timebar(elem, elements, msg, tstart, h);
   end
end

% disp(maxnbrs);

if (all(ishghandle(h, 'figure')))
   close(h);
end

% Compute Omnivariance
omnivar=prod(feats,1).^(1/dimensions);

% Compute Eigen Entropy
energy=sum(feats,1);
nfeats=bsxfun(@times,feats',1./energy')';
nfeats(isnan(nfeats))=1;
entropy=-sum(nfeats.*log(nfeats))./log(dimensions);
entropy(entropy==1)=0;

% Compute Eigen Structure
structure=1-entropy;

% Compute Fractional Anisotropy
evmeans=mean(feats,1);
evdevs=feats-repmat(evmeans,dimensions,1);
numer=dimensions.*sum(evdevs.^2,1);
denom=(dimensions-1).*sum(feats.^2,1);
fa=(numer./denom).^(1/2);

% Compute Anisotropy
ani=(feats(1,:)-feats(dimensions,:))./(feats(1,:));

% Compute Isotropy
iso=1-ani;

% Compute dimensional degree
dims=zeros(size(feats));
dims(1:dimensions-1,:)=(feats(1:dimensions-1,:)-feats(2:dimensions,:))./...
                       repmat(feats(1,:),dimensions-1,1);
dims(dimensions,:)=iso;

% Compute Dimensional Embedding
[~,embed]=max([dims;iso],[],1);

% Compute dimensional entropy
alpha=[dims;iso];
de=-sum(alpha.*log(alpha))./log(dimensions);

% Populate feature classes
%CLASSES=[feats;norms;dims;iso;ani;fa;entropy;structure;omnivar;embed;biases;ints];
CLASSES.features=feats;
CLASSES.normals=norms;
CLASSES.dimensionality=dims;
CLASSES.isotropy=iso;
CLASSES.anisotropy=ani;
CLASSES.FA=fa;
CLASSES.entropy=entropy;
CLASSES.structure=structure;
CLASSES.omnivariance=omnivar;
CLASSES.embedding=embed;
CLASSES.biases=biases;
CLASSES.intensity=ints;
CLASSES.de=de;

% st=zeros(size(QUERIES));
% 
% msg='Computing Normal Structure Tensors...';
% tstart=tic;
% h = timebar(1, elements, msg, tstart);
% 
% for elem=1:step:elements
%     
%    last=min(elem+step-1,elements);
%     
%    % Get the nearest neighbors of elem
%    if (userParams.neighbors <= 0)
%        % Perform a fixed radius search
%        NN=DATABASE.rnn(QUERIES(:,elem:last),...
%                        userParams.radius);
%    else
%        if (userParams.radius <= 0)
%            % Search unconstrained
%            NN=DATABASE.knn(QUERIES(:,elem:last),...
%                            userParams.neighbors);
%        else
%            % Search constrained to radius
%            NN=DATABASE.knn(QUERIES(:,elem:last),...
%                            userParams.neighbors,...
%                            'lim',userParams.radius);
%        end
%    end
% 
%    for nbr=1:size(NN,1)
%       normals=norms(:,NN{nbr});
% 
%       S=svd(normals',0);
%       st(:,elem+nbr-1)=S.*S./(length(NN{nbr})-1);
% 
% %       D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %       st(:,elem+nbr-1)=sort(D,'descend');
% 
% %       ST=normals*normals';
% 
% %       if (nbr == 1)
% %           D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %           disp('EIG:');
% %           disp(sqrt(D));
% %           
% %           S=svd(normals',0);
% %           S = S ./ sqrt(length(NN{nbr})-1);
% %           disp('SVD:');
% %           disp(abs(S));
% % 
% %       end
% 
%    end
% 
%    if(toc > 1)
%       tic;
%       h = timebar(elem, elements, msg, tstart, h);
%    end
% end
% 
% if (all(ishghandle(h, 'figure')))
%    close(h);
% end

% CLASSES = QUERIES;

end

%
% PARSEINPUTS    Support function to parse inputs into userParams structure
function [userParams] = parseInputs(varargin)

userParams = struct('radius', 0, 'neighbors', 0, 'counts', [], 'steps', []);

if length(varargin) == 1 || ~isnumeric(varargin{2})
    value = varargin{1};

    if (isscalar(value)  && ...
        isreal(value)    && ...
        value >= 5)
       userParams.neighbors = fix(value);
    else
       error('Damkjer:InvalidCount', ...
             ['Number of Neighbors must be a real valued positive '...
              'integer greater or equal to 5: ' num2str(value)]);
    end

    varargin(1) = [];
end

% Parse the Property/Value pairs
if rem(length(varargin), 2) ~= 0
   error('Damkjer:PropertyValueNotPair', ...
         ['Additional arguments must take the form of Property/Value '...
          'pairs']);
end

propertyNames = {'neighbors', 'radius', 'counts', 'steps'};

while ~isempty(varargin)
   property = varargin{1};
   value    = varargin{2};

   % If the property has been supplied in a shortened form, lengthen it
   iProperty = find(strncmpi(property, propertyNames, length(property)));

   if isempty(iProperty)
      error('Damkjer:InvalidProperty', 'Invalid Property');
   elseif length(iProperty) > 1
      error('Damkjer:AmbiguousProperty', ...
            'Supplied shortened property name is ambiguous');
   end
   
   property = propertyNames{iProperty};

   switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
            userParams.neighbors = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Number of Neighbors must be a real valued positive '...
                   'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidRadius', ...
                  'Radius must be a real valued positive scalar');
         end
      case 'counts'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value >= 5))
            userParams.counts = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Counts must be a sorted vector of real valued positive '...
                   'integers greater or equal to 5']);
         end
      case 'steps'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value > 0))
            userParams.steps = value;
         else
            error('Damkjer:InvalidSteps', ...
                  ['Steps must be a sorted vector of real valued positive '...
                   'values']);
         end
   end

    varargin(1:2) = [];
end

if (~isempty(userParams.counts) && userParams.neighbors >= 5)
   error('Damkjer:MutExOpts', ...
         '''neighbors'' and ''counts'' options are mutually exclusive');
end

if (~isempty(userParams.steps) && userParams.radius > 0)
   error('Damkjer:MutExOpts', ...
         '''steps'' and ''radius'' options are mutually exclusive');
end

if (~isempty(userParams.counts) && ~isempty(userParams.steps))
   error('Damkjer:MutExOpts', ...
         '''steps'' and ''counts'' options are mutually exclusive');
end

if (userParams.neighbors <= 0 && ...
    userParams.radius <= 0 && ...
    isempty(userParams.counts) && ...
    isempty(userParams.steps))
   userParams.radius = 1;
end

end

