% LasPublicHeader   ASPRS LAS format public header block
%
% File:
%    LasPublicHeader.m
%
% Description:
%    This MATLAB class represents an ASPRS LAS version 1.2 file public header
%    block.
%
% Limitations:
%    None.
%
% Properties:
%    fileSignature          - The File Signature
%    fileSourceID           - The File Source ID
%    globalEncoding         - The File Global Encoding
%    projectID_GUID1        - The Globally Unique (Project) Identifier - Part 1
%    projectID_GUID2        - The Globally Unique (Project) Identifier - Part 2
%    projectID_GUID3        - The Globally Unique (Project) Identifier - Part 3
%    projectID_GUID4        - The Globally Unique (Project) Identifier - Part 4
%    systemIdentifier       - The System Identifier
%    generatingSoftware     - The Generating Software
%    headerSize             - The Header Size in Bytes
%    offsetToPointData      - The Offset to Point Data
%    nVLRs                  - The Number of Variable Length Records
%    pointDataFormatID      - The Point Data Format ID
%    pointDataRecordLength  - The Point Data Record Length
%    nPoints                - The Number of Point Records
%    nPointsByReturn        - The Number of Point Records by Return
%    xScaleFactor           - The X Coordinate Scale Factor
%    yScaleFactor           - The Y Coordinate Scale Factor
%    zScaleFactor           - The Z Coordinate Scale Factor
%    xOffset                - The X Coordinate Offset
%    yOffset                - The Y Coordinate Offset
%    zOffset                - The Z Coordinate Offset
%    maxX                   - The Maximum X Coordinate
%    minX                   - The Minimum X Coordinate
%    maxY                   - The Maximum Y Coordinate
%    minY                   - The Minimum Y Coordinate
%    maxZ                   - The Maximum Z Coordinate
%    minZ                   - The Minimum Z Coordinate
%    extraData              - The Extra Data
%    fileCreationDate       - The File Creation Date
%    fileCreationDayOfYear  - The File Creation Day of Year
%    fileCreationYear       - The File Creation Year
%    version                - The Version Number
%
% Methods:
%    [header] = LasPublicHeader(varargin)  - Constructor for LasPublicHeader
%                                            objects.
%    [header] = loadFrom(location)         - Load LAS file public header block
%                                            from a given location.
%               saveTo(location)           - Save LAS file public header block
%                                            to a given location.
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    http://asprs.org/a/society/committees/standards/asprs_las_format_v12.pdf
%
% See Also:
%    LasFile
%

% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

classdef (Sealed = true) LasPublicHeader
   properties (SetAccess = private)
      % The File Signature
      %
      % The file signature must contain the four characters "LASF", and
      % it is required by the LAS specification. These four characters
      % can be checked by user software as a quick look initial
      % determination of file type.
      fileSignature         = 'LASF'
   end
   
   properties
      % The File Source ID
      %
      % This field should be set to a value between 1 and 65535,
      % inclusive. A value of zero (0) is interpreted to mean that an ID
      % has not been assigned. In this case, processing software is free
      % to assign any valid number. Note that this scheme allows a LIDAR
      % project to contain up to 65535 unique sources. A source can be
      % considered an original flight line or it can be the result of
      % merge and/or extract operations.
      fileSourceID          = uint16(0)
      
      % The File Global Encoding
      %
      % This is a bit field used to indicate certain global properties
      % about the file. In LAS 1.2 (the version in which this field is
      % introduced), only the low bit is defined (this is the bit, that
      % if set, would have the unsigned integer yield a value of 1). This
      % bit field is defined as:
      %
      %   Bits | Field Name    | Description
      %   -----+---------------+-----------------------------------------
      %   0    | GPS Time Type | The meaning of GPS Time in the Point
      %        |               | Records
      %        |               | 0(not set) -> GPS Time in the point
      %        |               | record fields is GPS Week Time (the same
      %        |               | as previous versions of LAS)
      %        |               | 1 (set) -> GPS Time is standard GPS Time
      %        |               | (satellite GPS Time) minus 1 x 10^9. The
      %        |               | offset moves the time back to near zero
      %        |               | to improve floating point resolution.
      %   -----+---------------+-----------------------------------------
      %   1:15 | Reserved      | Must be set to zero.
      globalEncoding        = uint16(0)
      
      % The Globally Unique (Project) Identifier - Part 1
      %
      % The four fields that comprise a complete Globally Unique
      % Identifier (GUID) are now reserved for use as a Project
      % Identifier (Project ID). The field remains optional. The time of
      % assignment of the Project ID is at the discretion of processing
      % software. The Project ID should be the same for all files that
      % are associated with a unique project. By assigning a Project ID
      % and using a File Source ID (defined above) every file within a
      % project and every point within a file can be uniquely identified,
      % globally.
      projectID_GUID1       = uint32(0)
      
      % The Globally Unique (Project) Identifier - Part 2
      %
      % The four fields that comprise a complete Globally Unique
      % Identifier (GUID) are now reserved for use as a Project
      % Identifier (Project ID). The field remains optional. The time of
      % assignment of the Project ID is at the discretion of processing
      % software. The Project ID should be the same for all files that
      % are associated with a unique project. By assigning a Project ID
      % and using a File Source ID (defined above) every file within a
      % project and every point within a file can be uniquely identified,
      % globally.
      projectID_GUID2       = uint16(0)
      
      % The Globally Unique (Project) Identifier - Part 3
      %
      % The four fields that comprise a complete Globally Unique
      % Identifier (GUID) are now reserved for use as a Project
      % Identifier (Project ID). The field remains optional. The time of
      % assignment of the Project ID is at the discretion of processing
      % software. The Project ID should be the same for all files that
      % are associated with a unique project. By assigning a Project ID
      % and using a File Source ID (defined above) every file within a
      % project and every point within a file can be uniquely identified,
      % globally.
      projectID_GUID3       = uint16(0)
      
      % The Globally Unique (Project) Identifier - Part 4
      %
      % The four fields that comprise a complete Globally Unique
      % Identifier (GUID) are now reserved for use as a Project
      % Identifier (Project ID). The field remains optional. The time of
      % assignment of the Project ID is at the discretion of processing
      % software. The Project ID should be the same for all files that
      % are associated with a unique project. By assigning a Project ID
      % and using a File Source ID (defined above) every file within a
      % project and every point within a file can be uniquely identified,
      % globally.
      projectID_GUID4       = ''
      
      % The System Identifier
      %
      % The version 1.0 specification assumes that LAS files are
      % exclusively generated as a result of collection by a hardware
      % sensor. Version 1.1 recognizes that files often result from
      % extraction, merging or modifying existing data files. Thus System
      % ID becomes:
      %
      %   Generating Agent                       | System ID
      %   ---------------------------------------+-----------------------
      %   Hardware System                        | String identifying
      %                                          | hardware (e.g. "ALTM
      %                                          | 1210" or "ALS50"
      %   ---------------------------------------+-----------------------
      %   Merge of one or more files             | "MERGE"
      %   ---------------------------------------+-----------------------
      %   Modification of a single file          | "MODIFICATION"
      %   ---------------------------------------+-----------------------
      %   Extraction from one or more files      | "EXTRACTION"
      %   ---------------------------------------+-----------------------
      %   Reprojection, rescaling, warping, etc. | "TRANSFORMATION"
      %   ---------------------------------------+-----------------------
      %   Some other operation                   | "OTHER" or a string up
      %                                          | to 32 characters
      %                                          | identifying the
      %                                          | operation
      systemIdentifier      = 'OTHER'
      
      % The Generating Software
      %
      % This information is ASCII data describing the generating software
      % itself. This field provides a mechanism for specifying which
      % generating software package and version was used during LAS file
      % creation (e.g. "TerraScan V-10.8", "REALM V-4.2" and etc.). If
      % the character data is less than 32 characters, the remaining data
      % must be null.
      generatingSoftware    = ['MATLAB ' version]
      
      % The Header Size
      %
      % The size, in bytes, of the Public Header Block itself. In the
      % event that the header is extended by a software application
      % through the addition of data at the end of the header, the Header
      % Size field must be updated with the new header size. Extension of
      % the Public Header Block is discouraged; the Variable Length
      % Records should be used whenever possible to add custom header
      % data. In the event a generating software package adds data to the
      % Public Header Block, this data must be placed at the end of the
      % structure and the Header Size must be updated to reflect the new
      % size.
      headerSize            = uint16(227)
      
      % The Offset to Point Data
      %
      % The actual number of bytes from the beginning of the file to the
      % first field of the first point record data field. This data
      % offset must be updated if any software adds data from the Public
      % Header Block or adds/removes data to/from the Variable Length
      % Records.
      offsetToPointData     = uint32(227)
      
      % The Number of Variable Length Records
      % This field contains the current number of Variable Length
      % Records. This number must be updated if the number of Variable
      % Length Records changes at any time.
      nVLRs                 = uint32(0)
      
      % The Point Data Format ID
      %
      % The point data format ID corresponds to the point data record
      % format type. LAS 1.2 defines types 0, 1, 2 and 3.
      pointDataFormatID     = uint8(0)
      
      % The Point Data Record Length
      %
      % The size, in bytes, of the Point Data Record.
      pointDataRecordLength = uint16(20)
      
      % The Number of Point Records
      %
      % This field contains the total number of point records within the
      % file.
      nPoints               = uint32(0)
      
      % The Number of Points by Return
      %
      % This field contains an array of the total point records per
      % return. The first unsigned long value will be the total number of
      % records from the first return, and the second contains the total
      % number for return two, and so forth up to five returns.
      nPointsByReturn       = zeros(5, 1, 'uint32')
      
      % The X Coordinate Scale Factor
      %
      % The scale factor fields contain a double floating point value
      % that is uysed to scale the corresponding X, Y, and Z long values
      % within the point records. The corresponding X, Y, and Z scale
      % factor must be multiplied by the X, Y, or Z point record value to
      % get the actual X, Y, or Z coordinate. For example, if the X, Y,
      % and Z coordinates are intended to have two decimal point values,
      % then each scale factor will contain the number 0.01.
      xScaleFactor          = double(0)
      
      % The Y Coordinate Scale Factor
      %
      % The scale factor fields contain a double floating point value
      % that is uysed to scale the corresponding X, Y, and Z long values
      % within the point records. The corresponding X, Y, and Z scale
      % factor must be multiplied by the X, Y, or Z point record value to
      % get the actual X, Y, or Z coordinate. For example, if the X, Y,
      % and Z coordinates are intended to have two decimal point values,
      % then each scale factor will contain the number 0.01.
      yScaleFactor          = double(0)
      
      % The Z Coordinate Scale Factor
      %
      % The scale factor fields contain a double floating point value
      % that is uysed to scale the corresponding X, Y, and Z long values
      % within the point records. The corresponding X, Y, and Z scale
      % factor must be multiplied by the X, Y, or Z point record value to
      % get the actual X, Y, or Z coordinate. For example, if the X, Y,
      % and Z coordinates are intended to have two decimal point values,
      % then each scale factor will contain the number 0.01.
      zScaleFactor          = double(0)
      
      % The X Coordinate Offset
      %
      % The offset fields should be used to set the overall offset for
      % the point records. In general these numbers will be zero, but for
      % certain cases the resolution of the point data may not be large
      % enough for a given projection system. However, it should always
      % be assumed that these numbers are used. So to scale a given X
      % from the point record, take the point record X multiplied by the
      % X scale factor, and then add the X offset.
      %
      % X_coordinate = (X_record * X_scale) + X_offset
      xOffset               = double(0)
      
      % The Y Coordinate Offset
      %
      % The offset fields should be used to set the overall offset for
      % the point records. In general these numbers will be zero, but for
      % certain cases the resolution of the point data may not be large
      % enough for a given projection system. However, it should always
      % be assumed that these numbers are used. So to scale a given Y
      % from the point record, take the point record Y multiplied by the
      % Y scale factor, and then add the Y offset.
      %
      % Y_coordinate = (Y_record * Y_scale) + Y_offset
      yOffset               = double(0)
      
      % The Z Coordinate Offset
      %
      % The offset fields should be used to set the overall offset for
      % the point records. In general these numbers will be zero, but for
      % certain cases the resolution of the point data may not be large
      % enough for a given projection system. However, it should always
      % be assumed that these numbers are used. So to scale a given Z
      % from the point record, take the point record Z multiplied by the
      % Z scale factor, and then add the Z offset.
      %
      % Z_coordinate = (Z_record * Z_scale) + Z_offset
      zOffset               = double(0)
      
      % The Maximum X Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      maxX                  = double(0)
      
      % The Minimum X Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      minX                  = double(0)
      
      % The Maximum Y Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      maxY                  = double(0)
      
      % The Minimum Y Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      minY                  = double(0)
      
      % The Maximum Z Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      maxZ                  = double(0)
      
      % The Minimum Z Coordinate
      %
      % The max and min data fields are the actual unscaled extents of
      % the LAS point file data, specified in the coordinate system of
      % the LAS data.
      minZ                  = double(0)
      
      % The Extra Data
      %
      % The public header may be extended, though it is highly
      % discouraged. If it is extended, we have no way of knowing how to
      % interpret the extra data. We can simply pass it through, though.
      extraData = zeros(0,'uint8');
   end
   
   properties (Access = private)
      % The File Creation Date
      %
      % The date on which the file was created expressed as a serial date
      % number.
      pFileCreationDate
      
      % The Major Version Number
      %
      % The version number consists of a major and minor field. The major
      % and minor fields combine to form the number that indicates the
      % format number of the current specification itself. For example,
      % specification number 1.2 (this version) would contain 1 in the
      % major field and 2 in the minor field.
      pVersionMajor          = uint8(1)
      
      % The Minor Version Number
      %
      % The version number consists of a major and minor field. The major
      % and minor fields combine to form the number that indicates the
      % format number of the current specification itself. For example,
      % specification number 1.2 (this version) would contain 1 in the
      % major field and 2 in the minor field.
      pVersionMinor          = uint8(2)
   end
   
   properties (Dependent)
      % The File Creation Date
      %
      % The date on which the file was created expressed as a date
      % string.
      fileCreationDate
      
      % The File Creation Day of Year
      %
      % The day, expressed as an unsigned short, on which this file was
      % created. Day is computed as the Greenwhich Mean Time (GMT) day.
      % January 1 is considered day 1.
      fileCreationDayOfYear
      
      % The File Creation Year
      %
      % The year, expressed as a four digit number, in which the file was
      % created.
      fileCreationYear
      
      % The Version Number
      %
      % The version number consists of a major and minor field. The major
      % and minor fields combine to form the number that indicates the
      % format number of the current specification itself. For example,
      % specification number 1.2 (this version) would contain 1 in the
      % major field and 2 in the minor field.
      version
   end
   
   methods
      function header = LasPublicHeader(varargin)
         % Constructor for LasFile public header block objects
         %
         % LasPublicHeader objects may be constructed either by using a
         % no-argument (default) constructor or by passing a file
         % descriptor or file name.
         switch (nargin)
            case 0
               header.pFileCreationDate = floor(datenum(1970,1,1)+...
                  java.lang.System.currentTimeMillis/...
                  (1e3*86400));
            case 1
               header = header.loadFrom(varargin{1});
            otherwise
               error('LasPublicHeader:UnexpectedInputs',...
                     'Unexpected number of inputs encountered.');
         end
      end
      
      function theDate = get.fileCreationDate(header)
         theDate = datestr(header.pFileCreationDate);
      end
      
      function theDOY = get.fileCreationDayOfYear(header)
         theDOY = header.pFileCreationDate-...
                  datenum(...
                     str2double(...
                        datestr(header.pFileCreationDate,'yyyy')),1,1)+1;
      end
      
      function theYear = get.fileCreationYear(header)
         theYear = uint16(...
                      str2double(...
                         datestr(header.pFileCreationDate,'yyyy')));
      end
      
      function theVersion = get.version(header)
         theVersion = [num2str(header.pVersionMajor) '.'...
                       num2str(header.pVersionMinor)];
      end
      
      function header = loadFrom(header, location)
         % Load the LAS file public header from the given location.
         
         if (ischar(location))
            % Assume single argument is file name
            [fid, msg]=fopen(location, 'r');
            
            if (fid < 0)
               error('LasPublicHeader:FileError',msg);
            end
         elseif (isnumeric(location))
            % Assume single argument is file ID
            fid = location;
            frewind(fid);
         else
            error('LasPublicHeader:InitError',...
                  'Unknown argument initializer');
         end
         
         % Check the file signature to make sure we have a LAS file
         if (~strcmp(sscanf(char(fread(fid,4,'uchar=>uchar')'),'%c'),...
                     header.fileSignature))
            error('LasPublicHeader:InvalidFile',...
                  'File does not appear to be a valid LAS file.');
         end
         
         header.fileSourceID = fread(fid,1,'uint16=>uint16');
         header.globalEncoding = fread(fid,1,'uint16=>uint16');
         header.projectID_GUID1 = fread(fid,1,'uint32=>uint32');
         header.projectID_GUID2 = fread(fid,1,'uint16=>uint16');
         header.projectID_GUID3 = fread(fid,1,'uint16=>uint16');
         header.projectID_GUID4 = ...
            sscanf(char(fread(fid,8,'uchar=>uchar')'),'%c');
         header.pVersionMajor = fread(fid,1,'uchar=>uchar');
         header.pVersionMinor = fread(fid,1,'uchar=>uchar');
         header.systemIdentifier= ...
            sscanf(char(fread(fid,32,'uchar=>uchar')'),'%c');
         header.generatingSoftware = ...
            sscanf(char(fread(fid,32,'uchar=>uchar')'),'%c');
         dayOfYear = fread(fid,1,'uint16=>double');
         year = fread(fid,1,'uint16=>double');
         header.pFileCreationDate = datenum(year,1,dayOfYear);
         header.headerSize = fread(fid,1,'uint16=>uint16');
         header.offsetToPointData = fread(fid,1,'uint32=>uint32');
         header.nVLRs = fread(fid,1,'uint32=>uint32');
         header.pointDataFormatID = fread(fid,1,'uint8=>uint8');
         header.pointDataRecordLength = fread(fid,1,'uint16=>uint16');
         header.nPoints = fread(fid,1,'uint32=>uint32');
         header.nPointsByReturn = fread(fid,5,'uint32=>uint32');
         header.xScaleFactor = fread(fid,1,'double=>double');
         header.yScaleFactor = fread(fid,1,'double=>double');
         header.zScaleFactor = fread(fid,1,'double=>double');
         header.xOffset = fread(fid,1,'double=>double');
         header.yOffset = fread(fid,1,'double=>double');
         header.zOffset = fread(fid,1,'double=>double');
         header.maxX = fread(fid,1,'double=>double');
         header.minX = fread(fid,1,'double=>double');
         header.maxY = fread(fid,1,'double=>double');
         header.minY = fread(fid,1,'double=>double');
         header.maxZ = fread(fid,1,'double=>double');
         header.minZ = fread(fid,1,'double=>double');
         
         if (header.headerSize > 227)
            header.extraData = fread(fid,header.headerSize - 227,'uint8=>uint8');
         end
         
         if (ischar(location))
            % Close the file if it was opened  here.
            fclose(fid);
         end
      end
      
      function saveTo(header, location)
         % Save the LAS file public header to the given location.
         
         if (ischar(location))
            % Assume single argument is file name
            [fid, msg]=fopen(location, 'w');
            
            if (fid < 0)
               error('LasPublicHeader:FileError',msg);
            end
         elseif (isnumeric(location))
            % Assume single argument is file ID
            fid = location;
         else
            error('LasPublicHeader:InitError',...
               'Unknown argument initializer');
         end
         
         fprintf(fid,'%4s',header.fileSignature);
         fwrite(fid,header.fileSourceID,'uint16');
         fwrite(fid,header.globalEncoding,'uint16');
         fwrite(fid,header.projectID_GUID1,'uint32');
         fwrite(fid,header.projectID_GUID2,'uint16');
         fwrite(fid,header.projectID_GUID3,'uint16');

         if (length(header.projectID_GUID4) > 8)
            fwrite(fid,header.projectID_GUID4(1:8),'char');
         else
            GUID4=[header.projectID_GUID4...
                   repmat(char(0),1,8-numel(header.projectID_GUID4))];
            fwrite(fid,GUID4,'char');
         end
         
         fprintf(fid,'%c',header.pVersionMajor);
         fprintf(fid,'%c',header.pVersionMinor);

         if (length(header.systemIdentifier) > 32)
             fwrite(fid,header.systemIdentifier(1:32),'char');
         else
            sysID=[header.systemIdentifier...
                   repmat(char(0),1,32-...
                   numel(header.systemIdentifier))];
            fwrite(fid,sysID,'char');
         end
         
         if (length(header.generatingSoftware) > 32)
            fwrite(fid,header.generatingSoftware(1:32),'char');
         else
            genSoft=[header.generatingSoftware...
                     repmat(char(0),1,32-...
                     numel(header.generatingSoftware))];
            fwrite(fid,genSoft,'char');
         end
         
         fwrite(fid,header.fileCreationDayOfYear,'uint16');
         fwrite(fid,header.fileCreationYear,'uint16');
         
         fwrite(fid,header.headerSize,'uint16');
         fwrite(fid,header.offsetToPointData,'uint32');
         
         fwrite(fid,header.nVLRs,'uint32');
         fwrite(fid,header.pointDataFormatID,'uint8');
         fwrite(fid,header.pointDataRecordLength,'uint16');
         fwrite(fid,header.nPoints,'uint32');
         fwrite(fid,header.nPointsByReturn,'uint32');
         fwrite(fid,header.xScaleFactor,'double');
         fwrite(fid,header.yScaleFactor,'double');
         fwrite(fid,header.zScaleFactor,'double');
         fwrite(fid,header.xOffset,'double');
         fwrite(fid,header.yOffset,'double');
         fwrite(fid,header.zOffset,'double');
         fwrite(fid,header.maxX,'double');
         fwrite(fid,header.minX,'double');
         fwrite(fid,header.maxY,'double');
         fwrite(fid,header.minY,'double');
         fwrite(fid,header.maxZ,'double');
         fwrite(fid,header.minZ,'double');
         fwrite(fid,header.extraData,'uint8');
         
         if (ischar(location))
            % Close the file if it was opened  here.
            fclose(fid);
         end
      end
   end
end
