//*****************************************************************************
// FILE:        ClassHandle.h
//
//    Copyright (C)  2012 Kristian L. Damkjer.
//
// DESCRIPTION: This class is intended to be used to wrap C++ objects for
//              transport across mex function calls in MATLAB.
//
// LIMITATIONS: To preserve the lifetime guarantees of the pointer, default
//              construction, copy construction and assignment are
//              explicitly disallowed.
//
//              The class handle owns the pointer it contains. Thus, when
//              the class handle is destroyed, the pointer is deleted.
//
//              Class handles are forced to be allocated on the heap by
//              using the named constructor idiom. Class handles should be
//              destroyed using the destroyHandleTo template function.
//
// SOFTWARE HISTORY:
//> 2012-OCT-08  K. Damkjer
//               Initial Coding.
//> 2013-JUN-24  K. Damkjer
//               Clarified Error ID.
//<
//*****************************************************************************
#ifndef ClassHandle_HEADER
#define ClassHandle_HEADER

#ifndef EXPORT
   #if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
      #define EXPORT
   #else
      #define EXPORT export
   #endif
#endif

#include <string>

// Fix "wide char" definition for older versions of MATLAB. This must be placed
// after other includes and before the mex.h include.
#if MATLAB_MAJOR <= 7 && MATLAB_MINOR <= 10 && defined(_CHAR16T)
   #define CHAR16_T
#endif

#include "mex.h"

namespace Damkjer
{
//*****************************************************************************
// FUNCTION: ptrAsMat
//*****************************************************************************
template<typename BASE>
mxArray* ptrAsMat(BASE*);
//> This function reinterprets the pointer to the base object as a MEX index
//  and encapsulates the result in a MATLAB matrix object to persist between
//  MEX function calls.
//<

//*****************************************************************************
// CLASS: ClassHandle
//*****************************************************************************
template<typename BASE>
class ClassHandle
{
public:
   ~ClassHandle();
   //> The destructor. This method is also responsible for invoking mexUnock to
   //  allow calling MEX files to be cleared after memory management tasks have
   //  completed.
   //<
   
   const BASE& obj() const { return *thePointer; }
   //> This method returns the object encapsulated by the ClassWrapper.
   //<
   
   bool isValid() const;
   //> This method returns true if and only if the ClassHandle object passes
   //  validity checks. This method prevents the arbitrary reinterpretation of
   //  non-ClassWrapper objects.
   //<

   friend mxArray* ptrAsMat<BASE>(BASE*);

private:
   ClassHandle(BASE*);
   //> We want to force objects to be created using the ptrAsMat named
   //  constructor idiom. This also forces ClassHandles to be allocated on
   //  the heap instead of on the stack. This method is responsible for 
   //  invoking mexLock to prevent MATLAB from arbitrarily clearing memory 
   //  managed by the wrapper.
   //<

   // Explicitly disable "default" constructors.
   ClassHandle();
   ClassHandle(const ClassHandle<BASE>&);
   ClassHandle<BASE>& operator=(const ClassHandle<BASE>&);

   BASE* thePointer;
   //> The pointer to the encapsulated object.
   //<
   
   std::string theName;
   //> The type ID of the base type. Used for validation.
   //<
   
   uint32_T theSignature;
   //> The file signature. Used for validation.
   //<
};

//*****************************************************************************
// FUNCTION: ptrAsMat
//*****************************************************************************
template<typename BASE>
mxArray* ptrAsMat(BASE* ptr)
{
   mxArray* mat = mxCreateNumericMatrix(1, 1, mxINDEX_CLASS, mxREAL);
   *((mwIndex*)mxGetData(mat)) =
                     reinterpret_cast<mwIndex>(new ClassHandle<BASE>(ptr));
   return mat;
}

//*****************************************************************************
// FUNCTION: matAsObj
//*****************************************************************************
template<typename BASE>
const BASE& matAsObj(const mxArray* mat)
{
   if (mxGetNumberOfElements(mat) != 1    ||
       mxGetClassID(mat) != mxINDEX_CLASS ||
       mxIsComplex(mat))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsObj:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }
   
   ClassHandle<BASE>* handle =
         reinterpret_cast<ClassHandle<BASE>*>(*((mwIndex*)mxGetData(mat)));

   if (!(handle->isValid()))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsObj:invalidHandle",
                        "Handle not valid.");
   }

   return handle->obj();
}

//*****************************************************************************
// FUNCTION: destroyHandleTo
//*****************************************************************************
template<typename BASE>
void destroyHandleTo(const mxArray* mat)
{
   if (mxGetNumberOfElements(mat) != 1    ||
       mxGetClassID(mat) != mxINDEX_CLASS ||
       mxIsComplex(mat))
   {
      mexErrMsgIdAndTxt("Damkjer:destroyHandleTo:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }
   
   ClassHandle<BASE>* handle =
         reinterpret_cast<ClassHandle<BASE>*>(*((mwIndex*)mxGetData(mat)));

   if (!(handle->isValid()))
   {
      mexErrMsgIdAndTxt("Damkjer:destroyHandleTo:invalidHandle",
                        "Handle not valid.");
   }

   delete handle;   
}
}

#if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
#include "ClassHandle.cpp"
#endif

#endif
