//*****************************************************************************
// FILE:        VpTree.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: This class is an implementation of the vantage point tree
//              data structure described by Peter Yianilos in "Data
//              Structures and Algorithms for Nearest Neighbor Search in
//              General Metric Spaces".
//
//              This structure includes the enhancement to collapse subtrees
//              near leaves into a single node. This change reduces overall
//              storage and some recursion overhead, but may make queries
//              less efficient if the leaf size is chosen to be too large.
//
// LIMITATIONS: The tree is currently coded as nearly the simplest version
//              of the proposed structure. This yields close to the fastest
//              construct time with potential limitations on query speed.
//
//              Extra effort is spent on finding a "good" pivot for large
//              trees with more effort devoted as the tree coverage grows
//              (currently determined by examining sqrt(size) elements for
//              pivot). The thought is that selection of a better pivot
//              carries more weight in queries as tree coverage increases.
//
//              The first proposed enhancement would be to track tighter
//              bounds on the branch sub-spaces. We currently only track
//              the median partition which provides only a loose upper
//              bound on the inner partition and a tight lower bound on the
//              outer partition. For a slight storage overhead, we can
//              instead track the tight lower and upper bounds for each
//              branch. Investing in this enhancement would allow for
//              earlier pruning of branches at the cost of increased
//              storage and construct time.
//
//              The second proposed enhancement would be to track distances
//              to ancestral pivots. This change potentially incurs a
//              signficant storage overhead, but allows for further
//              explicit pruning or inclusion optimizations for range
//              queries.
//
//              The final proposed enhancement would be to increase fan-out
//              of the tree by establishing multiple partition boundaries,
//              or by modifying the structure into an MVP tree.
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef VpTree_HEADER
#define VpTree_HEADER

#ifndef EXPORT
   #if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
      #define EXPORT
   #else
      #define EXPORT export
   #endif
#endif

#include "Util/SpatialIndexing/Metrics/Metrics.h"

#include <vector>
#include <deque>
#include <set>
#include <queue>
#include <limits>

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree
//*****************************************************************************
EXPORT template<typename T = std::vector<double>,
         double (*DISTANCE)(const T&, const T&) = euclidean<T> >
class VpTree
{
public:
   VpTree();
      //> The default constructor creates a null tree.
      //<

   template<typename CONTAINER>
   VpTree(const CONTAINER&, const std::size_t& leafCapacity=8);
      //> Create a VpTree from a (probably unsorted) container of items.
      //<
    
   virtual ~VpTree();
      //> Free resources allocated by this tree.
      //<
   
   std::pair<std::deque<std::size_t>, std::deque<double> > knn(
           const T&,
           const std::size_t&,
           const double limit=std::numeric_limits<double>::max()) const;
      //> Perform a k nearest neighbor search on the tree returning the
      //  indices of and distances to the k nearest neighbors.
      //<
   
   std::pair<std::deque<std::size_t>, std::deque<double> > rnn(
           const T&,
           const double) const;
      //> Perform a fixed radius nearest neighbor search on the tree
      //  returning the indices of and distances to the neighbors in the
      //  fixed radius.
      //<
    
private:
   VpTree(const VpTree&);
      //> Disable copy construction for VpTree.
      //<
    
   class Item;
      //> The VpsTree::Item class provides a simple container to capture a
      //  database element along with its index and ancestral pivot
      //  history.
      //<

   class ResultsCandidate;
      //> The VpTree::ResultsCandidate class provides a private
      //  representation of search result candidates to be employed in an
      //  intermediate priority queue of candidate results.
      //<
    
   class Node;
      //> The VpTree::Node class provides the private definition of
      //  general VpTree nodes.
      //<
    
   class Internal;
      //> The VpTree::Internal class provides the private definition of
      //  internal VpTree nodes.
      //<
    
   class Leaf;
      //> The VpTree::Node class provides the private definition of
      //  VpTree leaf nodes.
      //<

   friend class VpTree<T, DISTANCE>::Node;
   friend class VpTree<T, DISTANCE>::Internal;
   friend class VpTree<T, DISTANCE>::Leaf;

   Node* makeTree(const std::size_t&, const std::size_t&);
      //> Set this tree's root to be the root of the tree created from the
      //  argument set of metric-space elements.
      //<
    
   std::size_t selectVp(const std::size_t&, const std::size_t&) const;
      //> Select a vantage point in the range between the provided indices.
      //<
    
   std::set<std::size_t> randomSample(const std::size_t&,
                                      const std::size_t&) const;
      //> Select a random sample in the range between the provided indices.
      //<

   Node* theRoot;
      //> The VpTree root is the starting point for all queries against the
      //  database represented by the VpTree.
      //<
    
   std::deque<Item> theItems;
      //> The items are the set of metric-space elements managed by the
      //  VpTree paired with their original indices in the unordered
      //  database.
      //<
   
   std::size_t theLeafCapacity;
      //>
      //<
};

}

#if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
#include "VpTree.cpp"
#endif

#endif
