% VpTree   Vantage Point Tree spatial indexing data structure
%
% File:
%    VpTree.m
%
% Description:
%    This MATLAB class represents a simple vantage-point tree.
%
% Limitations:
%    See VpTree.h for complete description of limitations.
%
% Properties:
%    theTree  - Brief description
%
% Methods:
%        [tree] = VpTree(varargin)                 - Brief description
%                 delete(tree)                     - Brief description
%    [idx,dist] = knn(this, queries, k, varargin)  - Brief description
%    [idx,dist] = rnn(this, data, radius)          - Brief description
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    None.
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%

classdef (Sealed = true) VpTree < handle
   
   properties (Hidden=true, Access = private)
      theTree
      theCleanUpObj
   end
   
   methods
      function this = VpTree(varargin)
         % This function builds the index structure for the database.
         
         if (nargin ~= 1)
            error('VpTree:InvalidArgumentList',...
                  'VpTree must be constructed with a set of observations');
         end
         
         this.theTree = VpTreeAPI('create', varargin{1});
      end

      function delete(this)
         % This function frees memory on delete (clear) of the VpTree.
         VpTreeAPI('destroy', this.theTree);
      end
      
      function [idx,dist] = knn(this, queries, k, varargin)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         
         userParams = struct('limit', realmax);
         
         % Parse the Property/Value pairs
         if rem(length(varargin), 2) ~= 0
            error('Damkjer:PropertyValueNotPair', ...
                  ['Additional arguments must take the form of'...
                   'Property/Value pairs']);
         end
         
         propertyNames = {'limit'};
         
         while ~isempty(varargin)
            property = varargin{1};
            value    = varargin{2};
            
            % If the property has been supplied in a shortened form,
            % lengthen it
            iProperty = find(strncmpi(property, propertyNames, ...
                                      length(property)));
            
            if isempty(iProperty)
               error('Damkjer:InvalidProperty', 'Invalid Property');
            elseif length(iProperty) > 1
               error('Damkjer:AmbiguousProperty', ...
                     'Supplied shortened property name is ambiguous');
            end
            
            property = propertyNames{iProperty};
            
            switch property
               case 'limit'
                  if (isscalar(value) && isnumeric(value) && ...
                      isreal(value) && value > 0)
                     userParams.limit = value;
                  else
                     error('Damkjer:InvalidLimit', ...
                           'Limit must be a real valued positive scalar');
                  end
            end
            
            varargin(1:2) = [];
         end
         
         if (nargout == 1)
             idx = VpTreeAPI('search_kann', this.theTree, queries, k, userParams.limit);
         else
             [idx,dist] = VpTreeAPI('search_kann', this.theTree, queries, k, userParams.limit);
         end
      end
      
      function [idx,dist] = rnn(this, queries, radius)
         % Perform a fixed-radius nearest neighbor search on the
         % database with the set of queries.
         
         if (nargout == 1)
             idx = VpTreeAPI('search_frann', this.theTree, queries, radius);
         else
             [idx,dist] = VpTreeAPI('search_frann', this.theTree, queries, radius);
         end
      end
   end
end

