% [OUT]=BSCB_INPAINT(IN,...) Bartalmio et al image inpainting.
%   Performs image in-painting as described by Bertlamio, Sapiro, Caselles, and 
%   Ballester (2000).
%
%   Copyright 2012 Kristian L. Damkjer
%
%   Software History:
%      2012-AUG-29   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Updated to remove mis-interprestation of paper and add pre-processing
%         step.
%
function J=bscb_inpaint(I,varargin)

% Parse inputs
p = inputParser;
defaultPaintSteps = 15;
defaultSmoothSteps = 2;
defaultTimeSlice = 0.2;
defaultIters = 300;

addRequired(p, 'I', @isfloat);
addOptional(p, 'PaintSteps', defaultPaintSteps, @isnumeric);
addOptional(p, 'SmoothSteps', defaultSmoothSteps, @isnumeric);
addOptional(p, 'Iters', defaultIters, @isnumeric);
addOptional(p, 'TimeSlice', defaultTimeSlice, @isfloat);

% Establish default parameters
parse(p,I,varargin{:});
I=p.Results.I;
A=p.Results.PaintSteps;
B=p.Results.SmoothSteps;
T=p.Results.Iters;
t=p.Results.TimeSlice;

%Determine Omega (the inpaint region)
mask=zeros(size(I));
mask(:)=isnan(I(:));

%P=I;
%P(isnan(I))=0.5;

P=bscb_preproc(I);

J=P;

% Boundary must be large enough to support gradient of laplace computation
SE=strel('arbitrary',ones(5));
[r,c]=ind2sub(size(I),find(imdilate(mask,SE)));
diff_mask=[r,c];

[r,c]=ind2sub(size(I),find(mask));
mask=[r,c];

% Pre-processing: the whole image undergoes anisotropic diffusion smoothing

P=bscb_diffuse(P,t);
J(isnan(I))=P(isnan(I));

% figure(1); imshow(P); axis equal ij;

h = waitbar(0,'In Painting...');

Li=zeros(size(P));

for pass=1:T;
    for i=1:A
        % Equation (7)
        L = laplace(P,diff_mask);

        for idx=1:size(diff_mask,1)
            x=diff_mask(idx,2);
            y=diff_mask(idx,1);
            Li(y,x)=L(idx);
        end
        
        % Equation (6)
%         [Lx,Ly] = sobel(Li, mask);
        [Lx,Ly] = grad(Li, mask);

        % Equation (8)
%         [Ix,Iy] = sobel(P, mask);
        [Ix,Iy] = grad(P, mask);
        [Nx,Ny] = isophote(Ix, Iy);

        [Ixb,Iyb] = bgrad(P, mask);
        [Ixf,Iyf] = fgrad(P, mask);

        for idx=1:size(mask,1)
           
            x=mask(idx,2);
            y=mask(idx,1);
            beta=Lx(idx)*Nx(idx)+Ly(idx)*Ny(idx);
            
            mxb=0;
            mxf=0;
            myb=0;
            myf=0;
            
            It = 0;
            
            if (beta > 0)
                mxb = min(0,Ixb(idx));
                mxf = max(0,Ixf(idx));
                myb = min(0,Iyb(idx));
                myf = max(0,Iyf(idx));
                
                It = beta*sqrt(mxb^2+mxf^2+myb^2+myf^2);

%                mxb = max(0,Ixb(idx));
%                mxf = min(0,Ixf(idx));
%                myb = max(0,Iyb(idx));
%                myf = min(0,Iyf(idx));
                
%                It = 0.5*(It + (-beta)*sqrt(mxb^2+mxf^2+myb^2+myf^2));
            else
                mxb = max(0,Ixb(idx));
                mxf = min(0,Ixf(idx));
                myb = max(0,Iyb(idx));
                myf = min(0,Iyf(idx));

                It = beta*sqrt(mxb^2+mxf^2+myb^2+myf^2);

%                mxb = min(0,Ixb(idx));
%                mxf = max(0,Ixf(idx));
%                myb = min(0,Iyb(idx));
%                myf = max(0,Iyf(idx));

%                It = 0.5*(It + (-beta)*sqrt(mxb^2+mxf^2+myb^2+myf^2));

            end
            
%             It=beta*sqrt(mxb^2+mxf^2+myb^2+myf^2);
            
            % Non-linear scaling per Bertalmio, Processing of Flat and
            % non-Flat Image Information on Arbitrary Manifolds Using
            % Partial Differential Equations, Thesis, University of
            % Minnesota, 2001. p. 66
            si=sign(It);

%             if (rand(1) < 0.0001)
%                 disp(['Lap : ' num2str(Lx(idx)) ' ' num2Str(Ly(idx))]);
%                 disp(['Beta: ' num2str(beta)]);
%                 disp(['Sign: ' num2str(si)]);
%             end
            
            P(y,x)=min(max(0,P(y,x)+t*si*sqrt(sqrt(si*It))),1);
            
            % Paper claims following is sufficient, however I found that
            % often it failed to propagate information into the inpainting
            % domain.
%             P(y,x)=min(max(0,P(y,x)+t*It),1);
        end
    end
    
    for j = 1:B
        % Paper claims 't' used for both inpaint and diffuse steps, but
        % Bertalmio thesis implementation suggests otherwise

%         P=bscb_diffuse(P,t,mask);
        P=bscb_diffuse(P,0.2,mask);
    end
    
    J(isnan(I))=P(isnan(I));
    
    if (mod(pass,10) == 0||pass==1||pass==T)
        f=figure(1);
        imshow(J);
        pause(0.01);
    end
    
    if (all(ishghandle(h, 'figure')))
        waitbar(pass/T,h,['In Painting...' num2str(pass) '/' num2str(T)]);
    else
        h = waitbar(pass/T,['In Painting...' num2str(pass) '/' num2str(T)]);
    end
end

close(f);
close(h);
