%FDIFF3 Numerical Third Partial.
%   [FXXX,FXXY,FXYY,FYYY] = FDIFF3(F) returns the numerical third partial
%   derivative of the matrix F. The spacing between points in each direction is
%   assumed to be one.
%
%   [FXXX,FXXY,FXYY,FYYY] = FDIFF3(F,IDXS), where IDXS is a 2-D array, computes
%   numerical third partial derivative at the locations specified by IDXS.
%
%   Copyright 2012 Kristian L. Damkjer
%
%   Software History:
%      2012-AUG-29   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Updated output sizes or indexed case.
%

function [Oxxx,Oxxy,Oxyy,Oyyy] = fdiff3( I,idxs )

if (nargin<2)
    IcP = zeros(size(I));
    Ipp = zeros(size(I));
    Icp = zeros(size(I));
    Inp = zeros(size(I));
    IPc = zeros(size(I));
    Ipc = zeros(size(I));
    Inc = zeros(size(I));
    INc = zeros(size(I));
    Ipn = zeros(size(I));
    Icn = zeros(size(I));
    Inn = zeros(size(I));
    IcN = zeros(size(I));

    IcP(3:end,:)        =I(1:end-2,:);
    Ipp(2:end,2:end)    =I(1:end-1,1:end-1);
    Icp(2:end,:)        =I(1:end-1,:);
    Inp(2:end,1:end-1)  =I(1:end-1,2:end);
    IPc(:,3:end)        =I(:,1:end-2);
    Ipc(:,2:end)        =I(:,1:end-1);
    Inc(:,1:end-1)      =I(:,2:end);
    INc(:,1:end-2)      =I(:,3:end);
    Ipn(1:end-1,2:end)  =I(2:end,1:end-1);
    Icn(1:end-1,:)      =I(2:end,:);
    Inn(1:end-1,1:end-1)=I(2:end,2:end);
    IcN(1:end-2,:)      =I(3:end,:);


    Oxxx=0.5.*(-IPc+2.*Ipc-2.*Inc+INc);
    Oxxy=0.5.*(Ipn-2.*Icn+Inn-Ipp+2.*Icp-Inp);
    Oxyy=0.5.*(Inp-2.*Inc+Inn-Ipp+2.*Ipc-Ipn);
    Oyyy=0.5.*(-IcP+2.*Icp-2.*Icn+IcN);
else
    Oxxx = zeros(size(idxs,1),1);
    Oxxy = zeros(size(idxs,1),1);
    Oxyy = zeros(size(idxs,1),1);
    Oyyy = zeros(size(idxs,1),1);

    for idx=1:size(idxs,1)
        x=idxs(idx,2);
        y=idxs(idx,1);
        
        xP=max(1,x-2);
        xp=max(1,x-1);
        xn=min(x+1,size(I,2));
        xN=min(x+2,size(I,2));
        yP=max(1,y-2);
        yp=max(1,y-1);
        yn=min(y+1,size(I,1));
        yN=min(y+2,size(I,1));

        IcP=I(yP,x);
        Ipp=I(yp,xp);
        Icp=I(yp,x);
        Inp=I(yp,xn);
        IPc=I(y,xP);
        Ipc=I(y,xp);
        Inc=I(y,xn);
        INc=I(y,xN);
        Ipn=I(yn,xp);
        Icn=I(yn,x);
        Inn=I(yn,xn);
        IcN=I(yN,x);
        
        Oxxx(idx)=0.5.*(-IPc+2.*Ipc-2.*Inc+INc);
        Oxxy(idx)=0.5.*(Ipn-2.*Icn+Inn-Ipp+2.*Icp-Inp);
        Oxyy(idx)=0.5.*(Inp-2.*Inc+Inn-Ipp+2.*Ipc-Ipn);
        Oyyy(idx)=0.5.*(-IcP+2.*Icp-2.*Icn+IcN);
    end
end
