//*****************************************************************************
// FILE:        Exception.cpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The class implementations for exception classes that parallel the STL
//    exception classes.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-JUL-27  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iostream>    // HASA std::ostream interface
#include <sstream>     // USES std::stringstream interfaces

#include "Exception.h" // ISA Damkjer::Exception implementation

namespace Damkjer
{

//*****************************************************************************
// Exception::what()
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
Util_API const char* Exception::what() const throw()
{
   if (theMsg.empty())
   {
      std::ostringstream oss;
      message(oss);
      theMsg = oss.str();
   }

   return theMsg.c_str();
}

//*****************************************************************************
// Exception::message(std::ostream&)
//>   Apply standard formatting to the exception verbose message stream.
//
//    @param os The output stream to receive the formatted message.
//    @return   The updated stream.
//<
//*****************************************************************************
Util_API std::ostream& Exception::message(std::ostream& os) const
{
   os << "-- " << type() << " --\n";

   insertReason(os);

   os << "\n(" << who()
      << ", \"" << file() << "\""
      << ", line " << line() << ")";

   return os;
}

//*****************************************************************************
// Exception::push(const std::string&, const std::string&, const int)
//>   Add a record to the exception stack trace.
//
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
Util_API void Exception::push(const std::string& who,
                              const std::string& file,
                              const int          line) const
{
   std::ostringstream os;
   os << who << " (\"" << file << "\"" << ", line " << line << ")";
   theStack.push_back(os.str());
}
 
//*****************************************************************************
// Exception::stackTrace(std::ostream&)
//>   Provide a stack trace for the exceptional condition.
//
//    @param os The output stream to receive the stack trace.
//    @return   The updated stream.
//<
//*****************************************************************************
Util_API std::ostream& Exception::stackTrace(std::ostream& os) const
{
   if (theStack.empty()) return os;

   os << "\nStack Trace:";

   for (std::vector<std::string>::const_iterator vsi = theStack.begin();
        vsi != theStack.end();
        ++vsi)
   {
      os << "\n" << *vsi;
   }

   os << "\n";

   return os;
}

//*****************************************************************************
// Exception::Exception
//>   Allocate memory and instantiate an Exception object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
Util_API
Exception::Exception(const std::string& why,
                     const std::string& who,
                     const std::string& file,
                     int                line)
   : theWhy(why)
   , theWho(who)
   , theFile(file)
   , theLine(line)
{
}

//*****************************************************************************
// Exception::insertReason
//>   Insert the reason for raising the exception into the output stream.
//
//    @param os The output stream to receive the formatted reason for raising
//              the exception.
//    @return   The updated stream.
//<
//*****************************************************************************
Util_API std::ostream&
Exception::insertReason(std::ostream& os)
const
{
   return os << "\n" << theWhy << "\n";
}

//*****************************************************************************
// LogicError::LogicError
//>   Allocate memory and instantiate an LogicError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
LogicError::LogicError(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : Exception(why, who, file, line)
{
}

//*****************************************************************************
// DomainError::DomainError
//>   Allocate memory and instantiate an DomainError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
DomainError::DomainError(const std::string& why,
                         const std::string& who,
                         const std::string& file,
                         int                line)
   : LogicError(why, who, file, line)
   , std::domain_error(why)
{
}

//*****************************************************************************
// DomainError::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char*
DomainError::what()
const throw()
{
   return Exception::what();
}

//*****************************************************************************
// InvalidArgument::InvalidArgument
//>   Allocate memory and instantiate an InvalidArgument object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
InvalidArgument::InvalidArgument(const std::string& why,
                                 const std::string& who,
                                 const std::string& file,
                                 int                line)
   : LogicError(why, who, file, line)
   , std::invalid_argument(why)
{
}

//*****************************************************************************
// InvalidArgument::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char*
InvalidArgument::what()
const throw()
{
   return Exception::what();
}

//*****************************************************************************
// LengthError::LengthError
//>   Allocate memory and instantiate an LengthError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
LengthError::LengthError(const std::string& why,
                         const std::string& who,
                         const std::string& file,
                         int                line)
   : LogicError(why, who, file, line)
   , std::length_error(why)
{
}

//*****************************************************************************
// LengthError::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char* LengthError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// OutOfRange::OutOfRange
//>   Allocate memory and instantiate an OutOfRange object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
OutOfRange::OutOfRange(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : LogicError(why, who, file, line)
   , std::out_of_range(why)
{
}

//*****************************************************************************
// OutOfRange::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char* OutOfRange::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// RethrowError::RethrowError
//>   Allocate memory and instantiate an RethrowError object with details.
//
//    This constructor is used to wrap a std::exception as a Damkjer::Exception
//    for the purpose of rethrowing. The what() portion of the std::exception
//    is reported in the what() of the RethrowError.
//
//    @param thrown The original thrown std::exception to be re-thrown.
//    @param who    The module to add to the call stack.
//    @param file   The file containing the module.
//    @param line   The line number where the exception was encountered.
//<
//*****************************************************************************
RethrowError::RethrowError(const std::exception& thrown,
                           const std::string& who,
                           const std::string& file,
                           int line)
   : DomainError((*thrown.what() != '\0' ? thrown.what()
                                         : typeid(thrown).name()),
                 who, file, line)
{
}

//*****************************************************************************
// RuntimeError::RuntimeError
//>   Allocate memory and instantiate an RuntimeError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
RuntimeError::RuntimeError(const std::string& why,
                           const std::string& who,
                           const std::string& file,
                           int                line)
   : Exception(why, who, file, line)
{
}

//*****************************************************************************
// RangeError::RangeError
//>   Allocate memory and instantiate an RangeError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
RangeError::RangeError(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : RuntimeError(why, who, file, line)
   , std::range_error(why)
{
}

//*****************************************************************************
// RangeError::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char* RangeError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// OverflowError::OverflowError
//>   Allocate memory and instantiate an OverflowError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
OverflowError::OverflowError(const std::string& why,
                             const std::string& who,
                             const std::string& file,
                             int                line)
   : RuntimeError(why, who, file, line)
   , std::overflow_error(why)
{
}

//*****************************************************************************
// OverflowError::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char* OverflowError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// UnderflowError::UnderflowError
//>   Allocate memory and instantiate an UnderflowError object with details.
//
//    @param why  The reason for raising the exception.
//    @param who  The module to add to the call stack.
//    @param file The file containing the module.
//    @param line The line number where the exception was encountered.
//<
//*****************************************************************************
UnderflowError::UnderflowError(const std::string& why,
                               const std::string& who,
                               const std::string& file,
                               int                line)
   : RuntimeError(why, who, file, line)
   , std::underflow_error(why)
{
}

//*****************************************************************************
// UnderflowError::what
//>   Provide additional information about the exceptional condition.
//
//    @return The detailed error message.
//<
//*****************************************************************************
const char* UnderflowError::what() const throw()
{
   return Exception::what();
}

}
