//*****************************************************************************
// FILE:        Exception.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definitions for exception classes that parallel the STL
//    exception classes.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2013-JUL-27  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef Damkjer_Exception_HEADER
#define Damkjer_Exception_HEADER

#include <typeinfo>       // USES typeid operator.
#include <exception>      // ISA extension to STL exception classes.
#include <stdexcept>      // ISA extension to STL logic and runtime errors.
#include <vector>         // HASA std::vector data member.
#include <string>         // HASA std::string data member.
#include <ostream>        // USES std::ostream::operator<<.

#include "Util/UtilAPI.h" // ISA contributor to the Util libraray API.

namespace Damkjer
{

//*****************************************************************************
// CLASS: Exception
//> Abstract base class for exception classes. Exceptions allow for detailed,
//  robust handling of errors and exceptional circumstances encountered in
//  programs. This class, and its derivatives, intnentionally mirror the C++
//  Standard Library exception classes. They also provide for standardized
//  message formatting and stack trace aggregation through convenience macros.
//
//  The implemented exceptions that parallel the C++ Standard Library are:
//
// - Exception                ->     std::exception
//    - LogicError            ->        std::logic_error
//       - DomainError        ->           std::domain_error
//       - InvalidArgument    ->           std::invalid_argument
//       - LengthError        ->           std::length_error
//       - OutOfRange         ->           std::out_of_range
//    - RuntimeError          ->        std::runtime_error
//       - RangeError         ->           std::range_error
//       - OverflowError      ->           std::overflow_error
//       - UnderflowError     ->           std::underflow_error
//<
//*****************************************************************************
class Exception
{
public:
   // Use the compiler-generated default constructor.
   // Util_API Exception();

   // Use the compiler-generated copy constructor.
   // Util_API Exception(const Exception&);

   Util_API virtual ~Exception() throw(){}
      //> Destroy the Exception object and perform clean-up activities.
      //<

   // Use the compiler-generated assignment operator.
   // Util_API Exception& operator=(const Exception&);

   Util_API virtual const char* what() const throw() = 0;
      //> Provide additional information about the exceptional condition.
      //<

   Util_API virtual std::ostream& message(std::ostream&) const;
      //> Apply standard formatting to the exception verbose message stream.
      //<

   Util_API void push(const std::string&,
                      const std::string&,
                      const int) const;
      //> Add a record to the exception stack trace.
      //<

   Util_API std::ostream& stackTrace(std::ostream&) const;
      //> Provide a stack trace for the exceptional condition.
      //<

   Util_API const std::string& type() const {
      if (theTypeID.empty())
      {
         theTypeID = typeid(*this).name();
      }
      return theTypeID;
   }
      //> Provide the type of the exception class.
      //
      //  @return The type of the exception class.
      //<

   Util_API const std::string& why()  const { return theWhy; }
      //> Provide additional information about the cause of the exceptional
      //  condition, if possible.
      //
      //  @return The verbose description for the cause of the exception.
      //<

   Util_API const std::string& who()  const { return theWho; }
      //> Identify the module that raised the exception.
      //
      //  @return The module that raised the exception
      //<

   Util_API const std::string& file() const { return theFile; }
      //> Identify the file containing the code emitting the exception.
      //
      //  @return The file containing the code emitting the exception.
      //<

   Util_API const int line() const { return theLine; }
      //> Identify the line number of the originating location for the
      //  exception.
      //
      //  @return The line number for the exception origin.
      //<

protected:
   Util_API Exception(const std::string&,
                      const std::string&,
                      const std::string&,
                      int);
      //> Allocate memory and instantiate an Exception object with details.
      //<

   Util_API virtual std::ostream& insertReason(std::ostream&) const;
      //> Insert the reason for raising the exception into the output stream.
      //<

private:
   mutable std::vector<std::string> theStack;
      //> The call stack.
      //<

   mutable std::string theTypeID;
      //> The exception type.
      //<

   mutable std::string theMsg;
      //> The verbose message.
      //<

   std::string theWhy;
      //> The reason for raising the exception.
      //<

   std::string theWho;
      //> The originator of the exception.
      //<

   std::string theFile;
      //> The file containing the exception throw site.
      //<

   int theLine;
      //> The location of the exception throw.
      //<
};

//*****************************************************************************
// CLASS: LogicError
//>   An error raised due to faulty logic.
//
//    Logic errors are errors that, at least in theory, could be avoided by the
//    program; for example, by performing additional tests of function
//    arguments. Examples of such errors are a violation of logical
//    preconditions or a class invariant under the Design-by-Contract idiom.
//<
//*****************************************************************************
class LogicError : public Exception
{
public:
   Util_API LogicError(const std::string&,
                       const std::string&,
                       const std::string&,
                       int);
      //> Allocate memory and instantiate an LogicError object with details.
      //<

   Util_API virtual ~LogicError() throw() {}
      //> Destroy the LogicError object and perform clean-up activities.
      //<
};

//*****************************************************************************
// CLASS: DomainError
//>   An error raised due to an operation being undefined for an argument.
//
//    A domain error typically applies to mathematic functions where the domain
//    is the subset of values that can be accepted by the function. In this
//    case, an argument encountered outside of the valid subset of values is a
//    domain error.
//<
//*****************************************************************************
class DomainError : public LogicError,
                    public std::domain_error
{
public:
   Util_API DomainError(const std::string&,
                        const std::string&,
                        const std::string&,
                        int);
      //> Allocate memory and instantiate an DomainError object with details.
      //<

   Util_API virtual ~DomainError() throw() {}
      //> Destroy the DomainError object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: InvalidArgument
//>   An error raised due to an argument being rejected as valid input.
//
//    An invalid argument is a general class of exception that reflects a
//    violation of logical preconditions for a function.
//<
//*****************************************************************************
class InvalidArgument : public LogicError,
                        public std::invalid_argument
{
public:
   Util_API InvalidArgument(const std::string&,
                            const std::string&,
                            const std::string&,
                            int);
      //> Allocate memory and instantiate an InvalidArgument object with
      //  details.
      //<

   Util_API virtual ~InvalidArgument() throw() {}
      //> Destroy the InvalidArgument object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: LengthError
//>   An error raised due to an attempt to perform an action that would exceed
//    implementation defined length limits for an object.
//<
//*****************************************************************************
class LengthError : public LogicError,
                    public std::length_error
{
public:
   Util_API LengthError(const std::string&,
                        const std::string&,
                        const std::string&,
                        int);
      //> Allocate memory and instantiate an LengthError object with details.
      //<

   Util_API virtual ~LengthError() throw() {}
      //> Destroy the LengthError object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: OutOfRange
//>   An error raised due to an attempt to access elements outside of a defined
//    range.
//
//    A typical example of an out of range error is an invalid index for a
//    container element. 
//<
//*****************************************************************************
class OutOfRange : public LogicError,
                   public std::out_of_range
{
public:
   Util_API OutOfRange(const std::string&,
                       const std::string&,
                       const std::string&,
                       int);
      //> Allocate memory and instantiate an OutOfRange object with details.
      //<

   Util_API virtual ~OutOfRange() throw() {}
      //> Destroy the OutOfRange object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: RethrowError
//>   An error raised due to an unhandled exception thrown within a guarded
//    section.
//
//    Rethrow errors are used to capture the call trace between an exception
//    origin and its handling location.
//<
//*****************************************************************************
class RethrowError : public DomainError
{
public:
   Util_API RethrowError(const std::exception&,
                         const std::string&,
                         const std::string&,
                         int);
      //> Allocate memory and instantiate an RethrowError object with details.
      //<

   Util_API RethrowError(const Damkjer::Exception& thrown,
                         const std::string&        extraReason,
                         const std::string&        who,
                         const std::string&        file,
                         int                       line)
      : DomainError(thrown.why() + " " + extraReason, who, file, line)
   {
   }
      //> This constructor is used to add an extra reason to a
      //  Damkjer::Exception for the purpose of rethrowing.
      //
      //  @param thrown      The original thrown exception to be re-thrown.
      //  @param extraReason The additional reason for re-throwing.
      //  @param who         The module to add to the call stack.
      //  @param file        The file containing the module.
      //  @param line        The line number where the exception was
      //                     encountered.
      //<

   Util_API RethrowError(const std::string& why,
                         const std::string& who,
                         const std::string& file,
                         int                line)
      : DomainError(why, who, file, line)
   {
   }
      //> This constructor is used to wrap a string as a
      //  Damkjer::Exception for the purpose of rethrowing. The what()
      //  portion of the std::exception is reported in the what()
      //  of the RethrowError.
      //
      //  @param why  The reason for raising the exception.
      //  @param who  The module to add to the call stack.
      //  @param file The file containing the module.
      //  @param line The line number where the exception was encountered.
      //<

   Util_API RethrowError(const std::string& who,
                         const std::string& file,
                         int                line)
      : DomainError("Unknown error", who, file, line)
   {
   }
      //> This constructor is used to wrap an unknown exception as a
      //  Damkjer::Exception for the purpose of rethrowing. The message
      //  "Unknown error" is reported in the what() of the RethrowError.
      //
      //  @param who  The module to add to the call stack.
      //  @param file The file containing the module.
      //  @param line The line number where the exception was encountered.
      //<

   Util_API virtual ~RethrowError() throw() {}
      //> Destroy the RethrowError object and perform clean-up activities.
      //<
};

//*****************************************************************************
// CLASS: RuntimeError
//>   An error raised due to events beyond the scope of the program and only
//    detectable at run-time.
//
//    Runtime errors are errors that are not easily avoided by the program.
//<
//*****************************************************************************
class RuntimeError : public Exception
{
public:
   Util_API RuntimeError(const std::string&,
                         const std::string&,
                         const std::string&,
                         int);
      //> Allocate memory and instantiate an RuntimeError object with details.
      //<

   Util_API virtual ~RuntimeError() throw() {}
      //> Destroy the RuntimeError object and perform clean-up activities.
      //<
};

//*****************************************************************************
// CLASS: RangeError
//>   An error raised due to a defined, but unrepresentable result.
//
//    Range errors are typically used to report a floating-point value that
//    would be too large or small in magnitude to represent.
//
//    @note
//    Do not use this class to report out of range logical errors. For example,
//    an index out of bounds would not be a range error.
//<
//*****************************************************************************
class RangeError : public RuntimeError,
                   public std::range_error
{
public:
   Util_API RangeError(const std::string&,
                       const std::string&,
                       const std::string&,
                       int);
      //> Allocate memory and instantiate an RangeError object with details.
      //<

   Util_API virtual ~RangeError() throw() {}
      //> Destroy the RangeError object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: OverflowError
//>   An error raised due to a value growing too large resulting in an
//    arithmetic overflow.
//
//    Overflow errors are tpically used to report an integer computation that
//    would exceed the maximum representable value for the integer class.
//<
//*****************************************************************************
class OverflowError : public RuntimeError,
                      public std::overflow_error
{
public:
   Util_API OverflowError(const std::string&,
                          const std::string&,
                          const std::string&,
                          int);
      //> Allocate memory and instantiate an OverflowError object with details.
      //<

   Util_API virtual ~OverflowError() throw() {}
      //> Destroy the OverflowError object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

//*****************************************************************************
// CLASS: UnderflowError
//>   An error raised due to a value growing too small resulting in an
//    arithmetic underflow.
//
//    Underflow errors are tpically used to report an integer computation that
//    would exceed the minimum representable value for the integer class.
//<
//*****************************************************************************
class UnderflowError : public RuntimeError,
                       public std::underflow_error
{
public:
   Util_API UnderflowError(const std::string&,
                           const std::string&,
                           const std::string&,
                           int);
      //> Allocate memory and instantiate an UnderflowError object with
      //  details.
      //<

   Util_API virtual ~UnderflowError() throw() {}
      //> Destroy the UnderflowError object and perform clean-up activities.
      //<

   Util_API virtual const char* what() const throw();
      //> Provide additional information about the exceptional condition.
      //<
};

}

//*****************************************************************************
// FUNCTION: operator<<(std::ostream&, const Damkjer::Exception&)
//>   Insert a human-readable summary of the Damkjer::Exception into the output
//    stream.
//
//    @param  os The output stream to accept the error message.
//    @param  e  The Damkjer::Exception object to provide the error message.
//    @return    The updated output stream.
//<
//*****************************************************************************
Util_API inline std::ostream&
operator<<(std::ostream& os, const Damkjer::Exception& e)
{
   return os << e.what();
}

//*****************************************************************************
// MACRO: EXCEPTION_TRY
//>   Establish a guarded section for a module. The module name will be
//    appended to the stack trace for any unhandled exceptions. This macro
//    should be at the beginning of any module that may throw an exception.
//
//    @param MODULE_NAME The module identifier (and signature).
//<
//*****************************************************************************
#define EXCEPTION_TRY( MODULE_NAME ) \
static const char* const MODULE = MODULE_NAME; \
try \
{

//*****************************************************************************
// MACRO: EXCEPTION_RETHROW
//>   Intercept unhandled exceptions and append the current module to the stack
//    trace. This macro should be placed at the end of any module where an
//    unhandled exception may be encountered.
//<
//*****************************************************************************
#define EXCEPTION_RETHROW \
} \
catch (const Damkjer::Exception& e) \
{ \
   e.push(MODULE, __FILE__, __LINE__); \
   throw; \
} \
catch (const std::exception& e) \
{ \
   throw RethrowError(e, MODULE, __FILE__, __LINE__); \
} \
catch (...) \
{ \
   throw RethrowError(MODULE, __FILE__, __LINE__); \
}

//*****************************************************************************
// MACRO: EXCEPTION_CATCHALL
//>   Intercept exceptions immediately prior to throwing off the stack. Instead
//    of a hard crash, provide any exception messages and any stack trace
//    information available. Always return with a failure code to indicate
//    unsucessful program execution. This macro should be placed at the end of
//    all application main functions.
//<
//*****************************************************************************
#define EXCEPTION_CATCHALL \
} \
catch (const Damkjer::Exception& e) \
{ \
   e.push(MODULE, __FILE__, __LINE__); \
   std::cerr << e; \
   e.stackTrace(std::cerr); \
   return EXIT_FAILURE; \
} \
catch (const std::exception& e) \
{ \
   std::cerr << e.what() << std::endl; \
   return EXIT_FAILURE; \
} \
catch (...) \
{ \
   std::cerr << "Unknown exception" << std::endl; \
   return EXIT_FAILURE; \
}

#endif
