% ProjLinearUnitsGeoKey   Vantage Point Tree spatial indexing data structure
%
% File: ProjLinearUnitsGeoKey.m
%
% Description:
%    Provides the definition for GeoTIFF Projected Linear Units Keys.
%
% Limitations:
%    No known limitations.
%
% Properties:
%    code           - The code for the projected linear unit described by this
%                     key.
%    interpretation - The interpretation of the projected linear unit described
%                     by this key.
%
% Methods:
%        [this] = ProjLinearUnitsGeoKey(varargin) - Constructor
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    http://www.remotesensing.org/geotiff/spec/geotiffhome.html
%
% See Also:
%    None.
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2012-SEP-07   K. Damkjer
%       Initial Coding.
%

%******************************************************************************
% CLASS: ProjLinearUnitsGeoKey
%    GeoTIFF Projected Linear Units Key.  There are several different kinds of
%    units that may be used in geographically related raster data: linear
%    units, angular units, units of time (e.g. for radar-return), CCD-voltages,
%    etc. For this reason there will be a single, unique range for each kind of
%    unit, broken down into the following currently defined ranges:
%
%       Range          | Allocation
%       ---------------|--------------------------------------------------------
%       0              | undefined
%       [   1,  2000]  | Obsolete GeoTIFF codes
%       [2001,  8999]  | Reserved by GeoTIFF
%       [9000,  9099]  | EPSG Linear Units.
%       [9100,  9199]  | EPSG Angular Units.
%       32767          | user-defined unit
%       [32768, 65535] | Private User Implementations
%
%    Linear Unit Values (See the ESPG/POSC tables for definition):
%       Unit                        | Code
%       ----------------------------|------------------------------------------
%       Meter                       | 9001
%       Foot                        | 9002
%       Foot (US Survey)            | 9003
%       Foot (Modified American)    | 9004
%       Foot (Clarke)               | 9005
%       Foot (Indian)               | 9006
%       Link                        | 9007
%       Link (Benoit)               | 9008
%       Link (Sears)                | 9009
%       Chain (Benoit               | 9010
%       Chain (Sears)               | 9011
%       Yard (Sears)                | 9012
%       Yard (Indian)               | 9013
%       Fathom                      | 9014
%       International Nautical Mile | 9015
%******************************************************************************
classdef (Sealed = true) ProjLinearUnitsGeoKey < GeoKey
   properties
      % The code for the projected linear unit described by this key. The
      % default value, 0, indicates "undefined".
      code = 0
   end
   
   properties (Access = private, Constant = true)
      % The mapping of defined codes and interpretations.
      codes = containers.Map(...
         {9001,...
         9002,...
         9003,...
         9004,...
         9005,...
         9006,...
         9007,...
         9008,...
         9009,...
         9010,...
         9011,...
         9012,...
         9013,...
         9014,...
         9015},...
         {'Meter',...
         'Foot',...
         'Foot (US Survey)',...
         'Foot (Modified American)',...
         'Foot (Clarke)',...
         'Foot (Indian)',...
         'Link',...
         'Link (Benoit)',...
         'Link (Sears)',...
         'Chain (Benoit)',...
         'Chain (Sears)',...
         'Yard (Sears)',...
         'Yard (Indian)',...
         'Fathom',...
         'International Nautical Mile'});
   end
   
   properties (Dependent)
      % The interpretation of the projected linear unit described by this key.
      interpretation
   end
   
   methods
      %************************************************************************
      % ProjLinearUnitsGeoKey/ProjLinearUnitsGeoKey
      %************************************************************************
      function this = ProjLinearUnitsGeoKey(varargin)
         % The Constructor.  The constructor may be called with an argument
         % list to specify the code for the key. If no argument is provided,
         % the default constructor creates a key with "undefined" linear unit.
         %
         % Parameters:
         %    varargin - Variable-length input argument list. The argument list
         %               may contain up to a single argument. If an argument is
         %               provided, it will be checked against the list of known
         %               linear units. if the code is not recognized, an error
         %               will be thrown. Similarly, if the constructor is
         %               called with more than one argument, an error will be
         %               thrown.

         if (nargin == 0)
            this.code = 0;
         elseif (nargin == 1)
            if (~this.codes.isKey(varargin{1}))
                error('ProjLinearUnitsGeoKey:InitError',...
                      'Unknown Unit');
            end
            this.code = varargin{1};
         elseif (nargin > 1)
            error('ProjLinearUnitsGeoKey:InvalidArgument',...
                  'Too many arguments provided to constructor.');
         end
      end
      
      %************************************************************************
      % ProjLinearUnitsGeoKey/get.interpretation
      %************************************************************************
      function meaning = get.interpretation(this)
         % Reinterpret the key code in human-readable format.
         %
         % Returns:
         %    meaning - The interpretation of the code.
         
         meaning = this.codes(this.code);
      end
      
   end
end
   
   
