% VpTree   Vantage Point Tree spatial indexing data structure
%
% File: VpTree.m
%
% Description:
%    This MATLAB class represents a vantage-point tree data structure. The
%    class is derived from the MATLAB handle class to provide automated object
%    descruction on calls to "clear".
%
% Limitations:
%    See VpTree.h for complete description of limitations.
%
% Properties:
%    None.
%
% Methods:
%        [tree] = VpTree(varargin)                 - VP Tree Constructor
%                 delete(tree)                     - Release this object and
%                                                    free memory.
%    [idx,dist] = knn(this, queries, k, varargin)  - Perform a k-nearest
%                                                    neighbor search against
%                                                    the tree.
%    [idx,dist] = rnn(this, data, radius)          - Perform a fixed-radius
%                                                    nearest neighbor search
%                                                    against the tree.
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    VpTreeAPI MEX library.
%
% References:
%    Yianilos, P. N. (1993). Data Structures and Algorithms for Nearest
%       Neighbor Search in General Metric Spaces. In Proceedings of the Fourth
%       Annual ACM-SIAM Symposium on Discrete Algorithms (pp. 311321). SODA
%       93. Philadelphia, PA, USA: Society for Industrial and Applied
%       Mathematics. http://dl.acm.org/citation.cfm?id=313559.313789
%
% See Also:
%    None.
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%

%******************************************************************************
% CLASS: VpTree
%    Vantage Point Tree spatial indexing data structure. The class is derived
%    from the MATLAB handle class to provide automated object descruction on
%    calls to "clear".
%******************************************************************************
classdef (Sealed = true) VpTree < handle
  
   properties (Hidden=true, Access = private)
      % The C++ VP-tree object owned by this handle class.
      theTree
   end
   
   methods
      %************************************************************************
      % VpTree/VpTree
      %************************************************************************
      function this = VpTree(varargin)
         % Build the index structure for the database.
         %
         % Parameters:
         %    varargin - Variable-length input argument list. Currently, only a
         %               single argument is permitted in the argument list.
         %               Specifically, a set of point observations that will be
         %               spatially indexed by the tree structure.
         
         if (nargin ~= 1)
            error('VpTree:InvalidArgumentList',...
                  'VpTree must be constructed with a set of observations');
         end

         this.theTree = VpTreeAPI('create', varargin{1});
      end

      %************************************************************************
      % VpTree/delete
      %************************************************************************
      function delete(this)
         % This function frees memory on delete (clear) of the VpTree.
         VpTreeAPI('destroy', this.theTree);
      end

      %************************************************************************
      % VpTree/knn
      %************************************************************************
      function [idx,dist] = knn(this, queries, k, varargin)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %    queries  - The set of search locations about which to establish
         %               k-nearest neighborhoods.
         %
         %    k        - The number of nearest neighbors to find for the
         %               neighborhood.
         %
         %    varargin - Variable-length input argument list. The only defined
         %               argument at this time is 'limit' which is used to
         %               establish an upper bound on the radius of the
         %               k-nearest neighborhood.
         %
         % Returns:
         %    idx  - The indices for the k-nearest neighbors, sorted by
         %           distance to the query, for each of the input query points.
         %
         %    dist - The distance to each k-nearest neighbor, sorted by
         %           distance to the query, for each of the input query points.
         
         userParams = struct('limit', realmax);
         
         % Parse the Property/Value pairs
         if rem(length(varargin), 2) ~= 0
            error('Damkjer:PropertyValueNotPair', ...
                  ['Additional arguments must take the form of'...
                   'Property/Value pairs']);
         end
         
         propertyNames = {'limit'};
         
         while ~isempty(varargin)
            property = varargin{1};
            value    = varargin{2};
            
            % If the property has been supplied in a shortened form,
            % lengthen it
            iProperty = find(strncmpi(property, propertyNames, ...
                                      length(property)));
            
            if isempty(iProperty)
               error('Damkjer:InvalidProperty', 'Invalid Property');
            elseif length(iProperty) > 1
               error('Damkjer:AmbiguousProperty', ...
                     'Supplied shortened property name is ambiguous');
            end
            
            property = propertyNames{iProperty};
            
            switch property
               case 'limit'
                  if (isscalar(value) && isnumeric(value) && ...
                      isreal(value) && value > 0)
                     userParams.limit = value;
                  else
                     error('Damkjer:InvalidLimit', ...
                           'Limit must be a real valued positive scalar');
                  end
            end
            
            varargin(1:2) = [];
         end
         
         if (nargout == 1)
             idx = VpTreeAPI('search_kann', this.theTree,...
                             queries, k, userParams.limit);
         else
             [idx,dist] = VpTreeAPI('search_kann', this.theTree,...
                                    queries, k, userParams.limit);
         end
      end
      
      %************************************************************************
      % VpTree/rnn
      %************************************************************************
      function [idx,dist] = rnn(this, queries, radius)
         % Perform a fixed-radius nearest neighbor search on the
         % database with the set of queries.
         %
         % Parameters:
         %    queries - The set of search locations about which to establish
         %              k-nearest neighborhoods.
         %
         %    radius  - The maximum permitted distance to a nearest neighbor.
         %
         % Returns:
         %    idx  - The indices for the k-nearest neighbors, sorted by
         %           distance to the query, for each of the input query points.
         %
         %    dist - The distance to each k-nearest neighbor, sorted by
         %           distance to the query, for each of the input query points.
         if (nargout == 1)
             idx = VpTreeAPI('search_frann', this.theTree, queries, radius);
         else
             [idx,dist] = VpTreeAPI('search_frann', this.theTree,...
                                    queries, radius);
         end
      end
   end
end

