//*****************************************************************************
// FILE:        VpTreeAPI.cpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The application interface for working with VP Trees in MATLAB.
//<
//
// LIMITATIONS:
//>   All interfaces to the VP Tree objects from MATLAB must be routed through
//    this gateway file. This interface is required until and unless MATLAB
//    provides a mechanism for modifying locks on MEX files other than the
//    current file.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-19  K. Damkjer
//               Improved structure and documentation. Prevented double free of
//               VP Tree memory and infinite locking of MEX library.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************

#if _OPENMP
#include <omp.h>
#endif

#include "MATLAB/ClassHandle.h"
#include "Util/SpatialIndexing/VpTree/VpTree.h"                  // USES 

//***
// Fix "wide char" definition for older versions of MATLAB. This must be placed
// after other includes and before the mex.h include.
//***
#if (defined(MATLAB_MAJOR) && defined(MATLAB_MINOR))
   #if MATLAB_MAJOR <= 7 && MATLAB_MINOR <= 10 && defined(_CHAR16T)
      #define CHAR16_T
   #endif
#endif

#include "mex.h"

//*****************************************************************************
// NAMESPACE:
//>   Vantage Point Tree MEX API types, interfaces, and implementations.
//<
//*****************************************************************************
namespace
{

typedef double CoordT;
   //> The MATLAB VpTree point coordinate type.
   //<

typedef std::vector<CoordT> PointT;
   //> The MATLAB VpTree point type.
   //<

typedef std::deque<PointT> PointSetT;
   //> The MATLAB VpTree point set type.
   //<

typedef Damkjer::EuclideanDistance<PointT> MetricT;
   //> The MATLAB VpTree distance metric type.
   //<

typedef Damkjer::VpTree<MetricT> TreeT;
   //> The MATLAB VpTree type.
   //<

typedef Damkjer::ClassHandle<TreeT> HandleT;
   //> The VpTree MATLAB Class Handle type.
   //<

//*****************************************************************************
// FUNCTION: vpTreeCreate
//>   Create a VpTree for use in MATLAB.
//
//    New objects are wrapped by the ClassHandle template to ensure that locks
//    are managed correctly to avoid double frees of memory managed by this
//    class, to ensure objects have a life span separate from this function's
//    scope, and to provide convenience methods for passing the pointer to the
//    object between MATLAB and C++.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
vpTreeCreate(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs < 2 || nrhs > 2)
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeCreate:nargin",
                        "The vpTreeCreate function requires a single input.");
   }
   
   if (nlhs > 1)
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeCreate:nargout",
                        "The vpTreeCreate function requires a single output.");
   }

   //***
   // The single input required by vpTreeCreate is the collection of points to
   // be indexed.
   //***
   const mxArray* points=prhs[1];
   
   // Check to make sure that points are real-valued numerics
   if (mxIsSparse(points) ||
       mxGetNumberOfDimensions(points) != 2 ||
       mxIsComplex(points) ||
       !mxIsNumeric(points))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeCreate:prhs",
                        "Input to vpTreeCreate must be a full, 2-D matrix "
                        "of real-valued data representing N-dimensional "
                        "observations.");
   }

   // Attempt to make the VP Tree.
   const mwSize dims = mxGetM(points);
   const mwSize elems = mxGetN(points);
   
   double* data = mxGetPr(points);

   //***
   // Selection of data structure for the point database is not arbitrary.
   //
   // 1. Point coordinates are passed in as double-precision values. Preserve
   //    the data fidelity.
   // 2. Individual points are often low-dimensional. Keep the coordinates for
   //    a single point in contiguous memory by using a std::vector container.
   // 3. The point database can be quite large. Since VP Tree is an online data
   //    structure, allow references to points to occupy non-contiguous memory.
   //    This also speeds construction and destruction of the point database.
   //***
   PointSetT pointData(elems, PointT(dims));
   
   for (mwSize elem = elems; elem --> 0;)
   {
      for (mwSize dim = dims; dim --> 0;)
      {
         pointData[elem][dim]=(CoordT)(data[elem*dims+dim]);
      }
   }

   //***
   // Use the ClassHandle named constructor to provide a handle to the new VP
   // Tree object.
   //***
   try
   {
      plhs[0] = HandleT::createHandle(new TreeT(pointData, MetricT()));
   }
   catch (const std::exception& e)
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeCreate:initError", e.what());
   }
}

//*****************************************************************************
// FUNCTION: vpTreeDestroy
//>   Destroy the VpTree referenced by a ClassHandle object.
//
//    Destroying objects through the ClassHandle wrapper ensures memory leaks
//    are not introduced through this MEX function and allows the MEX library
//    to be unlocked once all objects have been destroyed and their memory
//    returned to the system.
//
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
inline
void
vpTreeDestroy(int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 2 || !mxIsNumeric(prhs[1]))
   {
       mexErrMsgIdAndTxt("Damkjer:vpTreeDestroy:varargin",
                         "The vpTreeDestroy function requires a single "
                         "input.");
   }

   // Free the VP Tree through the ClassHandle named destructor.
   HandleT::destroyHandle(prhs[1]);
}

//*****************************************************************************
// FUNCTION: vpTreeFRANN
//>   Perform a fixed-radius all nearest neighbor (FRANN) search on a VpTree.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
vpTreeFRANN(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs < 3 || nrhs > 4 || !mxIsNumeric(prhs[1]))
   {
       mexErrMsgIdAndTxt("Damkjer:vpTreeFRANN:varargin",
                         "Invalid number of input arguments.");
   }

   if (nlhs>2)
   {
       mexErrMsgIdAndTxt("Damkjer:vpTreeFRANN:varargout",
                         "Invalid number of output arguments.");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   const TreeT& tree = HandleT::handleReference(prhs[1]);

   // The second parameter should be the query points.
   const mxArray* queries=prhs[2];
    
   // Check to make sure that query points are real-valued numerics
   if (mxIsSparse(queries) ||
       mxGetNumberOfDimensions(queries) != 2 ||
       mxIsComplex(queries) ||
       !mxIsNumeric(queries))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeFRANN:prhs",
                        "Second input parameter to vpTreeFRANN must be a "
                        "full, 2-D matrix of real-valued data representing "
                        "multi-dimensional queries.");
   }

   // The third parameter should be the query radius.
   const mxArray* rData=prhs[3];

   // Check to make sure that radius is a real-valued numeric scalar.
   if (mxIsSparse(rData) ||
       mxGetNumberOfElements(rData) != 1 ||
       mxIsComplex(rData) ||
       !mxIsNumeric(rData))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeFRANN:prhs",
                        "Third input parameter to vpTreeFRANN must be a "
                        "real-valued scalar representing desired neighborhood "
                        "radius limit.");
   }

   //***
   // Get the query points.
   //
   // Selection of data structure for the query database is not arbitrary.
   //
   // 1. Point coordinates are passed in as double-precision values. Preserve
   //    the data fidelity.
   // 2. Individual points are often low-dimensional. Keep the coordinates for
   //    a single point in contiguous memory by using a std::vector container.
   // 3. The query database can be quite large. Allow references to points to
   //    occupy non-contiguous memory.
   //***
   const mwSize dims = mxGetM(queries);
   const mwSize elems = mxGetN(queries);
    
   double* data = mxGetPr(queries);
   PointSetT queryData(elems, PointT(dims));

   for (mwIndex elem = elems; elem --> 0;)
   {
      for (mwIndex dim = dims; dim --> 0;)
      {
         queryData[elem][dim]=(CoordT)(data[elem*dims+dim]);
      }
   }
   
   // Get the desired neighborhood radius limit.
   TreeT::DistT radius = (TreeT::DistT)(*(double*)mxGetData(rData));

   //***
   // The first output parameter holds the nearest neighbor indices. This
   // collection is represented as a cell array of vectors with a cell for each
   // query point.
   //***
   plhs[0] = mxCreateCellMatrix(elems, 1);

   //***
   // If desired, the distance from the query point to each of the nearest
   // neighbors can also be provided. This collection is represented as a cell 
   // array of vectors with a cell for each query point.
   //***
   if (nlhs==2)
   {
       plhs[1] = mxCreateCellMatrix(elems, 1);
   }

   //***
   // The following logic flows may seem counter-intuitive, but are structured
   // to intentionally avoid nested critical sections in parallelized code.
   //
   // Unfortunately, any calls to the MATLAB API must be treated as belonging
   // to a critical section since none of the API is thread-safe.
   //***

#if _OPENMP
   if (omp_get_num_threads() == 1)
   {
      omp_set_dynamic(1);
      omp_set_num_threads(omp_get_num_procs());
   }
#endif

   //***
   // The VP Tree data structure always provides results as a set of pairs of
   // indices and distances. There is no performance benefit to omitting the
   // distance computations since they are required for the search algorithm.
   //***
   std::deque<TreeT::SearchResultsT> results(queryData.size());

   //***
   // The first embarassingly parallelizable section simply searches for each
   // query point's neighbors in parallel. This is much simpler than attempting
   // to parallelize the search for a single point, and likely to yield 
   // superior results.
   //***
   #pragma omp parallel for
   for (int q = 0; q < queryData.size(); ++q)
   {
      unsigned int uq = static_cast<unsigned int>(q);
      results[uq] = tree.rnn(queryData[uq], radius);
   }

   //***
   // Allocating memory for the results to be passed back to MATLAB must take
   // place in a "critical section" since it involves exercising the MEX API.
   //***
   std::deque<mxArray*> nbr_idxs;
   std::deque<mwIndex*> point_idxs;
   
   for (int q = 0; q < queryData.size(); ++q)
   {
      unsigned int uq = static_cast<unsigned int>(q);
      mwSize neighbors = results[uq].first.size();

      nbr_idxs.push_back(mxCreateNumericMatrix(0, 0, mxINDEX_CLASS, mxREAL));
      mxSetM(nbr_idxs[uq], neighbors);
      mxSetN(nbr_idxs[uq], 1);
      mxSetData(nbr_idxs[uq], mxMalloc(sizeof(mwIndex)*neighbors*1));
      
      point_idxs.push_back((mwIndex*)mxGetData(nbr_idxs[uq]));
   }

   //***
   // Once memory has been allocated, the actual results can be populated in
   // parallel.
   //***
   #pragma omp parallel for
   for (int q = 0; q < queryData.size(); ++q)
   {  
      unsigned int uq = static_cast<unsigned int>(q);
      mwSize neighbors = results[uq].first.size();

      mwIndex* idxs = point_idxs[uq];
      
      for (mwIndex idx = neighbors; idx --> 0;)
      {
         idxs[idx]=results[uq].first[idx]+1;
      }
   }

   //***
   // Marking the data for return to MATLAB must take place in a "critical 
   // section" since it involves exercising the MEX API. This also changes
   // ownership and memory management responsibilities to MATLAB. We will not
   // free this data.
   //***   
   for (unsigned int q = 0; q < queryData.size(); ++q)
   {
      mxSetCell(plhs[0], q, nbr_idxs[q]);
   }

   // Repeat the "hand-off" to MATLAB for distance data, if it was requested.
   if (nlhs==2)
   {
      //***
      // Allocating memory for the results to be passed back to MATLAB must
      // take place in a "critical section" since it involves exercising the
      // MEX API.
      //***
      std::deque<mxArray*> nbr_dists;
      std::deque<double*>  point_dists;

      for (unsigned int q = 0; q < queryData.size(); ++q)
      {
         mwSize neighbors = results[q].first.size();

         nbr_dists[q] = mxCreateDoubleMatrix(0, 0, mxREAL);
         mxSetM(nbr_dists[q], neighbors);
         mxSetN(nbr_dists[q], 1);
         mxSetData(nbr_dists[q], mxMalloc(sizeof(double)*neighbors*1));
         
         point_dists.push_back(mxGetPr(nbr_dists[q]));
      }
      
      //***
      // Once memory has been allocated, the actual results can be populated in
      // parallel.
      //***
      #pragma omp parallel for
      for (int q = 0; q < queryData.size(); ++q)
      {
         unsigned int uq = static_cast<unsigned int>(q);
         mwSize neighbors = results[uq].first.size();
         
         double* dists = point_dists[uq];

         for (mwIndex idx = neighbors; idx --> 0;)
         {
            dists[idx]=results[uq].second[idx];
         }
      }

      //***
      // Marking the data for return to MATLAB must take place in a "critical 
      // section" since it involves exercising the MEX API. This also changes
      // ownership and memory management responsibilities to MATLAB. We will
      // not free this data.
      //***   
      for (unsigned int q = 0; q < queryData.size(); ++q)
      {
         mxSetCell(plhs[1], q, nbr_dists[q]);
      }
   }
}

//*****************************************************************************
// FUNCTION: vpTreeKANN
//>   Perform a k all nearest neighbor (KANN) search on a VpTree.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
vpTreeKANN(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 5 || !mxIsNumeric(prhs[1]))
   {
      mexErrMsgIdAndTxt("Damkjer:kannVpTree:varargin",
                        "Invalid number of arguments");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   const TreeT& tree = HandleT::handleReference(prhs[1]);

   // The second parameter should be the query points.
   const mxArray* queries=prhs[2];

   // Check to make sure that query points are real-valued numerics.
   if (mxIsSparse(queries) ||
       mxGetNumberOfDimensions(queries) != 2 ||
       mxIsComplex(queries) ||
       !mxIsNumeric(queries))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeKANN:prhs",
                        "Second input parameter to vpTreeKANN must be a full, "
                        "2-D matrix of real-valued data representing "
                        "multi-dimensional queries.");
   }

   // The third parameter should be the desired neighborhood cardinality.
   const mxArray* kData=prhs[3];

   // Check to make sure that cardinality is a real-valued numeric scalar.
   if (mxIsSparse(kData) ||
       mxGetNumberOfElements(kData) != 1 ||
       mxIsComplex(kData) ||
       !mxIsNumeric(kData))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeKANN:prhs",
                        "Third input parameter to vpTreeKANN must be an "
                        "real-valued scalar representing desired neighborhood "
                        "cardinality.");
   }

   // The fourth parameter should be the desired radius limit.
   const mxArray* rData=prhs[4];

   // Check to make sure that radius is a real-valued numeric scalar.
   if (mxIsSparse(rData) ||
       mxGetNumberOfElements(rData) != 1 ||
       mxIsComplex(rData) ||
       !mxIsNumeric(rData))
   {
      mexErrMsgIdAndTxt("Damkjer:vpTreeKANN:prhs",
                        "Fourth input parameter to vpTreeKANN must be an "
                        "real-valued scalar representing desired neighborhood "
                        "radius limit.");
   }

   //***
   // Get the query points.
   //
   // Selection of data structure for the query database is not arbitrary.
   //
   // 1. Point coordinates are passed in as double-precision values. Preserve
   //    the data fidelity.
   // 2. Individual points are often low-dimensional. Keep the coordinates for
   //    a single point in contiguous memory by using a std::vector container.
   // 3. The query database can be quite large. Allow references to points to
   //    occupy non-contiguous memory.
   //***
   const mwSize dims = mxGetM(queries);
   const mwSize elems = mxGetN(queries);

   double* data = mxGetPr(queries);
   PointSetT queryData(elems, PointT(dims));

   for (mwIndex elem = elems; elem --> 0;)
   {
      for (mwIndex dim = dims; dim --> 0;)
      {
         queryData[elem][dim]=(CoordT)(data[elem*dims+dim]);
      }
   }

   // Get the desired neighborhood cardinality.
   mwSize k = (mwSize)(*((double*)mxGetData(kData)));

   // Get the desired neighborhood radius limit.
   TreeT::DistT radius = (TreeT::DistT)(*(double*)mxGetData(rData));

   //***
   // The first output parameter holds the nearest neighbor indices. This
   // collection is represented as a cell array of vectors with a cell for each
   // query point.
   //***
   plhs[0] = mxCreateCellMatrix(elems, 1);

   //***
   // If desired, the distance from the query point to each of the nearest
   // neighbors can also be provided. This collection is represented as a cell 
   // array of vectors with a cell for each query point.
   //***
   if (nlhs==2)
   {
      plhs[1] = mxCreateCellMatrix(elems, 1);
   }

   //***
   // The following logic flows may seem counter-intuitive, but are structured
   // to intentionally avoid nested critical sections in parallelized code.
   //
   // Unfortunately, any calls to the MATLAB API must be treated as belonging
   // to a critical section since none of the API is thread-safe.
   //***

#if _OPENMP
   if (omp_get_num_threads() == 1)
   {
      omp_set_dynamic(1);
      omp_set_num_threads(omp_get_num_procs());
   }
#endif

   //***
   // The VP Tree data structure always provides results as a set of pairs of
   // indices and distances. There is no performance benefit to omitting the
   // distance computations since they are required for the search algorithm.
   //***
   std::deque<TreeT::SearchResultsT> results(queryData.size());

   //***
   // The first embarassingly parallelizable section simply searches for each
   // query point's neighbors in parallel. This is much simpler than attempting
   // to parallelize the search for a single point, and likely to yield 
   // superior results.
   //***
   #pragma omp parallel for
   for (int q = 0; q < queryData.size(); ++q)
   {
      unsigned int uq = static_cast<unsigned int>(q);
      results[uq] = tree.knn(queryData[uq], k, radius);
   }

   //***
   // Allocating memory for the results to be passed back to MATLAB must take
   // place in a "critical section" since it involves exercising the MEX API.
   //***
   std::deque<mxArray*> nbr_idxs;
   std::deque<mwIndex*> point_idxs;
   
   for (unsigned int q = 0; q < queryData.size(); ++q)
   {
      mwSize neighbors = results[q].first.size();

      nbr_idxs.push_back(mxCreateNumericMatrix(0, 0, mxINDEX_CLASS, mxREAL));
      mxSetM(nbr_idxs[q], neighbors);
      mxSetN(nbr_idxs[q], 1);
      mxSetData(nbr_idxs[q], mxMalloc(sizeof(mwIndex)*neighbors*1));
      
      point_idxs.push_back((mwIndex*)mxGetData(nbr_idxs[q]));
   }

   //***
   // Once memory has been allocated, the actual results can be populated in
   // parallel.
   //***
   #pragma omp parallel for
   for (int q = 0; q < queryData.size(); ++q)
   {  
      unsigned int uq = static_cast<unsigned int>(q);
      mwSize neighbors = results[uq].first.size();

      mwIndex* idxs = point_idxs[uq];
      
      for (mwIndex idx = neighbors; idx --> 0;)
      {
         idxs[idx]=results[uq].first[idx]+1;
      }
   }

   //***
   // Marking the data for return to MATLAB must take place in a "critical 
   // section" since it involves exercising the MEX API. This also changes
   // ownership and memory management responsibilities to MATLAB. We will not
   // free this data.
   //***   
   for (unsigned int q = 0; q < queryData.size(); ++q)
   {
      mxSetCell(plhs[0], q, nbr_idxs[q]);
   }

   // Repeat the "hand-off" to MATLAB for distance data, if it was requested.
   if (nlhs==2)
   {
      //***
      // Allocating memory for the results to be passed back to MATLAB must
      // take place in a "critical section" since it involves exercising the
      // MEX API.
      //***
      std::deque<mxArray*> nbr_dists;
      std::deque<double*>  point_dists;

      for (unsigned int q = 0; q < queryData.size(); ++q)
      {
         mwSize neighbors = results[q].first.size();

         nbr_dists[q] = mxCreateDoubleMatrix(0, 0, mxREAL);
         mxSetM(nbr_dists[q], neighbors);
         mxSetN(nbr_dists[q], 1);
         mxSetData(nbr_dists[q], mxMalloc(sizeof(double)*neighbors*1));
         
         point_dists.push_back(mxGetPr(nbr_dists[q]));
      }
      
      //***
      // Once memory has been allocated, the actual results can be populated in
      // parallel.
      //***
      #pragma omp parallel for
      for (int q = 0; q < queryData.size(); ++q)
      {
         unsigned int uq = static_cast<unsigned int>(q);
         mwSize neighbors = results[uq].first.size();
         
         double* dists = point_dists[uq];

         for (mwIndex idx = neighbors; idx --> 0;)
         {
            dists[idx]=results[uq].second[idx];
         }
      }

      //***
      // Marking the data for return to MATLAB must take place in a "critical 
      // section" since it involves exercising the MEX API. This also changes
      // ownership and memory management responsibilities to MATLAB. We will
      // not free this data.
      //***   
      for (unsigned int q = 0; q < queryData.size(); ++q)
      {
         mxSetCell(plhs[1], q, nbr_dists[q]);
      }
   }
}

}

//*****************************************************************************
// FUNCTION: mexFunction
//>   The MATLAB Executable Gateway Function.
//
//    Provides the application interface for working with VP Trees in MATLAB.
//    All functional interfaces are routed through this single function to
//    ensure locks and memory management are handled properly by MATLAB.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
mexFunction(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // This MEX function should always have at least one input parameter to
   // handle function dispatching through the MEX gateway.
   //***
   if (nrhs < 1)
   {
      mexErrMsgIdAndTxt("Damkjer:VpTreeAPI:nargin",
                        "VpTreeAPI requires at least one input which "
                        "specifies the operation to be performed.");
   }

   // The operation switch is simply a string.
   char* operation = mxArrayToString(prhs[0]);
   
   if (!operation)
   {
      mexErrMsgIdAndTxt("Damkjer:VpTreeAPI:InvalidOperation",
                        "Invalid mode supplied to VpTreeAPI.");
      return; // error should force return.
   }
   
   // Dispatch to helper functions. Err if operation is not recognized.
   if (!strcmp("create", operation))
   {
      vpTreeCreate(nlhs, plhs, nrhs, prhs);

   }
   else if (!strcmp("destroy", operation))
   {
      if (nlhs != 0)
      {
      mexErrMsgIdAndTxt("Damkjer:VpTreeAPI:nargout",
                        "Nothing to return.");
      }

      vpTreeDestroy(nrhs, prhs);
   }   
   else if (!strcmp("search_frann", operation))
   {
      vpTreeFRANN(nlhs, plhs, nrhs, prhs);
   }
   else if (!strcmp("search_kann", operation))
   {  
      vpTreeKANN(nlhs, plhs, nrhs, prhs);
   }
   else
   {
       mexErrMsgIdAndTxt("Damkjer:VpTreeAPI:UnknownOperation",
                         "Unrecognized mode provided to VpTreeAPI.");
   }

   // Prevent a slow memory leak.
   mxFree(operation);
}
