//*****************************************************************************
// FILE:        ChebyshevDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//> The public interface defintion for the Chebyshev distance metric functor.
//<
//
// LIMITATIONS:
//> No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef Damkjer_ChebyshevDistance_HEADER
#define Damkjer_ChebyshevDistance_HEADER

#if _MSC_VER
#pragma warning(push, 0)
#endif

#include <cmath>    // USES std::abs
#include <iterator> // USES container const_iterator interface

#if _MSC_VER
#pragma warning(pop)
#endif

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: ChebyshevDistance
//> The Chebyshev distance metric.
//
//  Distances between two points, x and y, in n-dimensional Chebyshev space are
//  calculated as follows:
//
//  <!--// d(x,y) = max_i(|x[i] - y[i]|)  //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \max_{1 \le i \le n}\left(\left|x_{i}-y_{i}\right|\right)
//  \f]
//
//  @note
//  The ChebyshevDistance functor must operate on an STL container class, or a
//  class that supports "duck" typing through a public typedef named
//  "value_type".
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT = typename PointT::value_type>
class ChebyshevDistance
{
public:

   typedef PointT  value_type;
      //> The point type for this functor.
      //<

   typedef ReturnT return_type;
      //> The return type for this functor.
      //<

   //***
   // Use compiler-generated constructor, copy-constructor and destructor.
   // ChebyshevDistance();
   // ChebyshevDistance(const ChebyshevDistance&);
   // ~ChebyshevDistance();
   //***

   //***
   // Use compiler-generated assignment operator.
   // ChebyshevDistance& operator=(const ChebyshevDistance&);
   //***

   ReturnT operator()(const PointT&, const PointT&) const;
      //> Perform the distance calculation.
      //<
};

//*****************************************************************************
// ChebyshevDistance::operator()(const PointT&, const PointT&)
//> Perform the distance calculation.
//
//  Distances between two points, x and y, in n-dimensional Chebyshev space are
//  calculated as follows:
//
//  <!--// d(x,y) = max_i(|x[i] - y[i]|)  //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \max_{1 \le i \le n}\left(\left|x_{i}-y_{i}\right|\right)
//  \f]
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//  @param x the first point.
//  @param y the second point.
//  @return the distance between the two argument points.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT>
inline ReturnT
ChebyshevDistance<PointT, ReturnT>::operator()(const PointT& x,
                                               const PointT& y)
const
{
   //***
   // For predictable data fidelity, perform computations with the same
   // fidelity as the point-type.
   //***
   typename PointT::value_type temp, my_max = 0;
      
   for (typename PointT::const_iterator xi = x.begin(), yi = y.begin();
        xi != x.end() && yi != y.end();
        ++xi, ++yi)
   {
      temp = std::abs(*xi - *yi);
      my_max = (temp > my_max) ? temp : my_max;
   }
      
   // Then static-cast to the desired return type.
   return static_cast<ReturnT>(my_max);
}

}

#endif
