//*****************************************************************************
// FILE:        EuclideanDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//> The public interface defintion for the Euclidean distance metric functor.
//<
//
// LIMITATIONS:
//> No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef Damkjer_EuclideanDistance_HEADER
#define Damkjer_EuclideanDistance_HEADER

#include <cmath>    // USES std::sqrt
#include <iterator> // USES container const_iterator interface

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: EuclideanDistance
//> The Euclidean distance metric.
//
//  Distances between two points, x and y, in n-dimensional Euclidean space are
//  calculated as follows:
//
//  <!--// d(x,y) = sqrt(sum_{i=0}^{n}((x[i] - y[i])^2)) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \sqrt{\sum_{i=0}^{n}\left(x_{i}-y_{i}\right)^{2}}
//  \f]
//
//  @note
//  The EuclideanDistance functor must operate on an STL container class, or a
//  class that supports "duck" typing through a public typedef named
//  "value_type".
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT = typename PointT::value_type>
class EuclideanDistance
{
public:
   typedef PointT  value_type;
      //> The point type for this functor.
      //<

   typedef ReturnT return_type;
      //> The return type for this functor.
      //<

   //***
   // Use compiler-generated constructor, copy-constructor and destructor.
   // EuclideanDistance();
   // EuclideanDistance(const EuclideanDistance&);
   // ~EuclideanDistance();
   //***

   //***
   // Use compiler-generated assignment operator.
   // EuclideanDistance& operator=(const EuclideanDistance&);
   //***

   ReturnT operator()(const PointT&, const PointT&) const;
      //> Perform the distance calculation.
      //<
};

//*****************************************************************************
// EuclideanDistance::operator()(const PointT&, const PointT&)
//> Perform the distance calculation.
//
//  Distances between two points, x and y, in n-dimensional Euclidean space are
//  calculated as follows:
//
//  <!--// d(x,y) = sqrt(sum_{i=0}^{n}((x[i] - y[i])^2)) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \sqrt{\sum_{i=0}^{n}\left(x_{i}-y_{i}\right)^{2}}
//  \f]
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//  @param x the first point.
//  @param y the second point.
//  @return the distance between the two argument points.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT>
inline ReturnT
EuclideanDistance<PointT, ReturnT>::operator()(const PointT& x,
                                               const PointT& y)
const
{
   //***
   // For predictable data fidelity, perform computations with the same
   // fidelity as the point-type.
   //***
   typename PointT::value_type sum = 0;

   for (typename PointT::const_iterator xi = x.begin(), yi = y.begin();
        xi != x.end() && yi != y.end();
        ++xi, ++yi)
   {
      sum += (*xi - *yi) * (*xi - *yi);
   }

   // Then static-cast to the desired return type.
   return static_cast<ReturnT>(std::sqrt(sum));
}

}

#endif
