//*****************************************************************************
// FILE:        ManhattanDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//> The public interface defintion for the Manhattan distance metric functor.
//<
//
// LIMITATIONS:
//> No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef Damkjer_ManhattanDistance_HEADER
#define Damkjer_ManhattanDistance_HEADER

#include <cmath>    // USES std::abs
#include <iterator> // USES container const_iterator interface

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: ManhattanDistance
//> The Manhattan distance metric.
//
//  Distances between two points, x and y, in n-dimensional Manhattan space are
//  calculated as follows:
//
//  <!--// d(x,y) = sum{i=0:n}(|x[i] - y[i]|) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \sum_{i=0}^{n}\left(\left|x_{i}-y_{i}\right|\right)
//  \f]
//
//  @note
//  The ManhattanDistance functor must operate on an STL container class, or a
//  class that supports "duck" typing through a public typedef named
//  "value_type".
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT = typename PointT::value_type>
class ManhattanDistance
{
public:
   typedef PointT  value_type;
      //> The point type for this functor.
      //<

   typedef ReturnT return_type;
      //> The return type for this functor.
      //<

   //***
   // Use compiler-generated constructor, copy-constructor and destructor.
   // ManhattanDistance();
   // ManhattanDistance(const ManhattanDistance&);
   // ~ManhattanDistance();
   //***

   //***
   // Use compiler-generated assignment operator.
   // ManhattanDistance& operator=(const ManhattanDistance&);
   //***

   ReturnT operator()(const PointT&, const PointT&) const;
      //> Perform the distance calculation.
      //<
};

//*****************************************************************************
// ManhattanDistance::operator()(const PointT&, const PointT&)
//> Perform the distance calculation.
//
//  Distances between two points, x and y, in n-dimensional Manhattan space are
//  calculated as follows:
//
//  <!--// d(x,y) = sum{i=0:n}(|x[i] - y[i]|) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \sum_{i=0}^{n}\left(\left|x_{i}-y_{i}\right|\right)
//  \f]
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//  @param x the first point.
//  @param y the second point.
//  @return the distance between the two argument points.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT>
inline ReturnT
ManhattanDistance<PointT, ReturnT>::operator()(const PointT& x,
                                               const PointT& y)
const
{
   //***
   // For predictable data fidelity, perform computations with the same
   // fidelity as the point-type.
   //***
   typename PointT::value_type sum = 0;
      
   for (typename PointT::const_iterator xi = x.begin(), yi = y.begin();
        xi != x.end() && yi != y.end();
        ++xi, ++yi)
   {
      sum += std::abs(*xi - *yi);
   }
      
   // Then static-cast to the desired return type.
   return static_cast<ReturnT>(sum);
}

}

#endif
