//*****************************************************************************
// FILE:        MinkowskiDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//> The public interface defintion for the Minkowski distance metric functor.
//<
//
// LIMITATIONS:
//> No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef Damkjer_MinkowskiDistance_HEADER
#define Damkjer_MinkowskiDistance_HEADER

#include <cmath>                      // USES std::abs and std::pow
#include <iterator>                   // USES container const_iterator interface

#include "Util/Exception/Exception.h" // USES Damkjer::DomainException interface

namespace Damkjer {

//*****************************************************************************
// FUNCTOR: MinkowskiDistance
//> The Minkowski distance metric.
//
//  Distances between two points, x and y, in n-dimensional Minkowski space are
//  calculated as follows:
//
//  <!--// d(x,y) = (sum_{i=0)^{n}((x[i] - y[i])^p))^(1/p) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \left(\sum_{i=0}^{n}\left(\left|x_{i}-
//                                 y_{i}\right|\right)^{p}\right)^{\frac{1}{p}}
//  \f]
//
//  @note
//  The MinkowskiDistance functor must operate on an STL container class, or a
//  class that supports "duck" typing through a public typedef named
//  "value_type".
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT = typename PointT::value_type>
class MinkowskiDistance
{
public:

   explicit MinkowskiDistance(typename PointT::value_type p = Euclidean);
      //> Intialize the Minkowski distance metric with a power value.
      //<

   //***
   // Use compiler-generated copy-constructor and destructor
   // MinkowskiDistance(const MinkowskiDistance&);
   // ~MinkowskiDistance(const MinkowskiDistance&);
   //***

   //***
   // Use compiler-generated assignment operator.
   // MinkowskiDistance& operator=(const MinkowskiDistance&);
   //***

   ReturnT operator()(const PointT& a, const PointT& b) const;
      //> Perform the distance calculation.
      //<

   typedef PointT  value_type;
      //> The point type for this functor.
      //<

   typedef ReturnT return_type;
      //> The return type for this functor.
      //<

private:
   //***
   // ENUMERATION: Special
   //>   List of pre-defined Minkowski metrics.
   //<
   //***   
   enum Special
   {
      Manhattan = 1, //> Manhattan distance is defined when p == 1.
                     //<
      Euclidean = 2  //> Euclidean distance is defined when p == 2.
                     //<
   };

   typename PointT::value_type thePower;
      //> The power value.
      //<
};

//*****************************************************************************
// MinkowskiDistance::MinkowskiDistance(PointT::value_type)
//> Intialize the Minkowski distance metric with a power value.
//
//  This constructor is the default constructor if no arguments are provided.
//  The default power value creates a Euclidean metric (p = 2).
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//  @param p the power value. Default to Euclidean (2).
//  @throw DomainError when p < 1.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT>
MinkowskiDistance<PointT, ReturnT>::MinkowskiDistance(
                                                 typename PointT::value_type p)
   : thePower(p)
{
   EXCEPTION_TRY("Damkjer::MinkowskiDistance<PointT, ReturnT>::"
                 "MinkowskiDistance");

   if (thePower < 1)
   {
      //***
      // In released code, there's not a whole lot that the user can do
      // about this problem. This is a logic error and should be prevented
      // through appropriate bounds checking on the input prior to
      // calling the constructor.
      //***
      std::ostringstream msg;

      msg << "Error encountered while defining a Minkowski distance "
          << "metric.\n\n"
          << "Unable to define a Minkowski distance metric with the "
          << "requested p-value: " << thePower << ". The Minkowski distance "
          << "must be defined with p >= 1 to describe a metric space.";

      throw DomainError(msg.str(),  MODULE, __FILE__, __LINE__);
   }

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// MinkowskiDistance::operator()(const PointT&, const PointT&)
//> Perform the distance calculation.
//
//  Distances between two points, x and y, in n-dimensional Minkowski space are
//  calculated as follows:
//
//  <!--// Easy to read format: //-->
//  <!--// d(x,y) = (sum_{i=0)^{n}((x[i] - y[i])^p))^(1/p) //-->
//
//  <!--// Pretty Doxygen (LaTeX) format: //-->
//  \f[
//     d\left(\bm{x},\bm{y}\right)=
//        \left(\sum_{i=0}^{n}\left(\left|x_{i}-
//                                 y_{i}\right|\right)^{p}\right)^{\frac{1}{p}}
//  \f]
//
//  @tparam PointT  the point type.
//  @tparam ReturnT the return type.
//  @param x the first point.
//  @param y the second point.
//  @return the distance between the two argument points.
//<
//*****************************************************************************
template<typename PointT, typename ReturnT>
inline ReturnT
MinkowskiDistance<PointT, ReturnT>::operator()(const PointT& x,
                                               const PointT& y)
const
{
   //***
   // For predictable data fidelity, perform computations with the same
   // fidelity as the point-type.
   //***
   typename PointT::value_type sum = 0;

   for (typename PointT::const_iterator xi = x.begin(), yi = y.begin();
        xi != x.end() && yi != y.end();
        ++xi, ++yi)
   {
      sum += std::pow(std::abs(*xi - *yi), thePower);
   }

   // Then static-cast to the desired return type.
   static const typename PointT::value_type ONE =
                                           static_cast<PointT::value_type>(1.);

   return static_cast<ReturnT>(std::pow(sum, ONE/thePower));
}

}

#endif
