//*****************************************************************************
// FILE:        VpTree_Branch.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for vantage-point tree branches.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************
#ifndef Damkjer_VpTree_Branch_HEADER
#define Damkjer_VpTree_Branch_HEADER

#if _OPENMP
#include <omp.h>
#endif

#include <algorithm>
#include <iostream>

#include "VpTree_Node.h" // ISA Damkjer::VpTree::Node

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree::Branch
//>   A vantage-point tree internal (non-terminal) node.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
class VpTree<MetricT>::Branch : public VpTree<MetricT>::Node
{
public:
   Branch();
      //> Instantiate a null node.
      //<
    
   Branch(VpTree*);
      //> Instantiate a node with a database reference.
      //<
    
   virtual ~Branch();
      //> Destruct the node and deallocate resources.
      //<
    
   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const;
      //> Accumulate the k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const;
      //> Accumulate the fixed-radius nearest neighbor results.
      //<

private:

   IndexT theIndex;
      //> The index of the item in the internal database.
      //<

   DistT theInnerLowerBound;
      //> The lower bound distance to elements on inner branch.
      //<

   DistT theInnerUpperBound;
      //> The uuper bound distance to elements on inner branch.
      //<

   DistT theOuterLowerBound;
      //> The lower bound distance to elements on outer branch.
      //<

   DistT theOuterUpperBound;
      //> The upper bound distance to elements on outer branch.
      //<

   Node* theInnerBranch;
      //> The inner branch partition containing elements closer than
      //  the median partition to this element.
      //<

   Node* theOuterBranch;
      //> The outer branch partition containing elements at or beyond
      //  the median partition from this element.
      //<

   friend class VpTree<MetricT>;
      //> Provide access to branch internals to the containing tree class.
      //<
};

//*****************************************************************************
// VpTree::Branch::Branch
//>   Instantiate a null node.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Branch::Branch()
   : Node()
   , theIndex(0)
   , theInnerLowerBound(0)
   , theInnerUpperBound(0)
   , theOuterLowerBound(0)
   , theOuterUpperBound(0)
   , theInnerBranch(0)
   , theOuterBranch(0)
{
}

//*****************************************************************************
// VpTree::Branch::Branch
//>   Instantiate a node with a database reference.
//
//    @tparam MetricT The metric search space.
//    @param  tree    The vantage point tree containing the item database.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Branch::Branch(VpTree<MetricT>* tree)
   : Node(tree)
   , theIndex(0)
   , theInnerLowerBound(0)
   , theInnerUpperBound(0)
   , theOuterLowerBound(0)
   , theOuterUpperBound(0)
   , theInnerBranch(0)
   , theOuterBranch(0)
{
}

//*****************************************************************************
// VpTree::Branch::~Branch
//>   Destruct the node and deallocate resources. Virtual to ensure proper node
//    destruction.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Branch::~Branch()
{
   delete theInnerBranch;
   delete theOuterBranch;
   theInnerBranch = 0;
   theOuterBranch = 0;
}

}

#include "VpTree_Branch.hpp"

#endif
