//*****************************************************************************
// FILE:        VpTree_Branch.hpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The template implementation for vantage-point tree branches.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************

#if _OPENMP
#include <omp.h>
#endif

#include <algorithm>
#include <iostream>

#include "VpTree_Branch.h"

namespace Damkjer
{

//*****************************************************************************
// VpTree::Branch::knn
//>   Accumulate the k nearest neighbor results.
//
//    @tparam        MetricT     The metric search space.
//    @param[in]     query       The query point to focus the search about.
//    @param[in]     k           The number of neighbors to identify.
//    @param[in,out] candidates  The set of candidate results.
//    @param[in,out] kth_closest The distance to the kth-closest item.
//<
//*****************************************************************************
template<typename MetricT>
void
VpTree<MetricT>::Branch::knn(const PointT& query,
                             const IndexT& k,
                             ResultsSetT& candidates,
                             DistT& kth_closest)
const
{
   if (!(this->theTree)) return;

   DistT distance = this->theTree->theMetric(
                                  this->theTree->theItems[theIndex].theElement,
                                  query);

   if (distance < kth_closest)
   {
      if (candidates.size() == k)
      {
         candidates.pop();
      }
      
      candidates.push(ResultsCandidate(theIndex, distance));
      
      if (candidates.size() == k)
      {
         kth_closest = candidates.top().theDistance;
      }
   }

   if (!(theInnerBranch || theOuterBranch))
   {
      return;
   }

   DistT middle = static_cast<DistT>
                  (0.5 * (theInnerUpperBound + theOuterLowerBound));

   if (distance < middle)
   {
      if (theInnerBranch &&
          (distance - theInnerUpperBound <= kth_closest) &&
          (theInnerLowerBound - distance <= kth_closest))


      if (theInnerBranch &&
          (distance <= theInnerUpperBound + kth_closest) &&
          (distance >= theInnerLowerBound - kth_closest))
      {
         theInnerBranch->knn(query, k, candidates, kth_closest);
      }
       
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - kth_closest) &&
          (distance <= theOuterUpperBound + kth_closest))
      {
         theOuterBranch->knn(query, k, candidates, kth_closest);
      }
   }
   else
   {
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - kth_closest) &&
          (distance <= theOuterUpperBound + kth_closest))
      {
         theOuterBranch->knn(query, k, candidates, kth_closest);
      }
       
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + kth_closest) &&
          (distance >= theInnerLowerBound - kth_closest))
      {
         theInnerBranch->knn(query, k, candidates, kth_closest);
      }
   }
}

//*****************************************************************************
// VpTree::Branch::rnn
//>   Accumulate the fixed-radius nearest neighbor results.
//
//    @tparam        MetricT    The metric search space.
//    @param[in]     query      The query point to focus the search about.
//    @param[in]     range      The radial distance used to bound the search.
//    @param[in,out] candidates The set of candidate results.
//<
//*****************************************************************************
template<typename MetricT>
void
VpTree<MetricT>::Branch::rnn(const PointT& query,
                             const DistT& range,
                             ResultsSetT& candidates)
const
{
   if (!(this->theTree)) return;
   
   DistT distance = this->theTree->theMetric(
                                  this->theTree->theItems[theIndex].theElement,
                                  query);

   if (distance <= range)
   {
      candidates.push(ResultsCandidate(theIndex, distance));
   }

   if (!(theInnerBranch || theOuterBranch))
   {
      return;
   }

   DistT middle = static_cast<DistT>
                  (0.5 * (theInnerUpperBound + theOuterLowerBound));

   if (distance < middle)
   {
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + range) &&
          (distance >= theInnerLowerBound - range))
      {
         theInnerBranch->rnn(query, range, candidates);
      }
       
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - range) &&
          (distance <= theOuterUpperBound + range))
      {
         theOuterBranch->rnn(query, range, candidates);
      }
   }
   else
   {
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - range) &&
          (distance <= theOuterUpperBound + range))
      {
         theOuterBranch->rnn(query, range, candidates);
      }
       
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + range) &&
          (distance >= theInnerLowerBound - range))
      {
         theInnerBranch->rnn(query, range, candidates);
      }
   }
}

}
