//*****************************************************************************
// FILE:        VpTree_Item.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for vantage-point tree items.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************
#ifndef Damkjer_VpTree_Item_HEADER
#define Damkjer_VpTree_Item_HEADER

#if _OPENMP
#include <omp.h>
#endif

#include <algorithm>
#include <iostream>

#include "VpTree.h"

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree::Item
//>   A simple container to capture a database element along with its index and
//    ancestral pivot history.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
class VpTree<MetricT>::Item
{
public:
   Item();
      //> The default constructor creates a blank item.
      //<

   //***
   // Use the compiler-generated copy constructor.
   // Item(const Item&);
   //***

   ~Item(){}
      //> The default destructor. Intentionally non-virtual since Node is
      //  a private inner class on VpTree.
      //<

   //***
   // Use the compiler-generated assignment operator.
   // Item& operator=(const Item&);
   //***

   bool operator<(const Item&) const;
      //> Less-than comparison for vantage point tree items.
      //<

private:
   IndexT theIndex;
      //> The index of the item in the input data set.
      //<

   PointT theElement;
      //> The database object
      //<
   
   DistT theDistance;
      //> The most recent ancestral pivot history distance for this item.
      //<

   friend class VpTree<MetricT>;
      //> Provide access to item internals to the containing tree class.
      //<
};

//*****************************************************************************
// VpTree::Item::Item()
//>   The default constructor creates a blank item.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Item::Item()
   : theIndex(0)
   , theElement(0)
   , theDistance(0)
{
}

//*****************************************************************************
// VpTree::Item::operator<(const Item&)
//>   Less-than comparison for vantage point tree items.
//
//    @note
//    In general, the other members of the comparison operator family should be
//    implemented. They are not in the case of vantage point tree items because
//    the less-than comparitor is only used in sorting semantics. The remaining
//    operators would thus be defined, but never used since items are private
//    to the vantage point tree.
//
//    @tparam MetricT The metric search space.
//    @param  other   The vantage point tree item for comparison against this
//                    item.
//    @return         true if and only if this item is closer to the query than
//                    the other item.
//<
//*****************************************************************************
template<typename MetricT>
inline
bool
VpTree<MetricT>::Item::operator<(const Item& other)
const
{
   return theDistance < other.theDistance;
}

}

#endif
