//*****************************************************************************
// FILE:        VpTree_Leaf.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for vantage-point tree leaves.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************
#ifndef Damkjer_VpTree_LeafNode_HEADER
#define Damkjer_VpTree_LeafNode_HEADER

#if _OPENMP
#include <omp.h>
#endif

#include <algorithm>
#include <iostream>

#include "VpTree_Node.h"

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree::Leaf
//>   A vantage-point tree terminal node.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
class VpTree<MetricT>::Leaf : public VpTree<MetricT>::Node
{
public:
   Leaf();
      //> Instantiate a null node.
      //<
    
   Leaf(VpTree*, const IndexT&, const IndexT&);
      //> Instantiate a leaf with a database reference that represents the
      //  items in the provided index range.
      //<
    
   virtual ~Leaf(){}
      //> Destruct the node and deallocate resources.
      //
      //    @tparam MetricT The metric search space.
      //<
    
   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const;
      //> Accumulate the k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const;
      //> Accumulate the fixed-radius nearest neighbor results.
      //<

private:
   IndexT theHead;
      //> The start of the index range represented by this leaf node.
      //<

   IndexT theTail;
      //> The end of the index range represented by this leaf node.
      //<
};

//*****************************************************************************
// VpTree::Leaf::Leaf
//>   Instantiate a null node.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Leaf::Leaf()
   : Node()
   , theHead(0)
   , theTail(0)
{
}

//*****************************************************************************
// VpTree::Leaf::Leaf
//>   Instantiate a leaf with a database reference that represents the items in
//    the provided index range.
//
//    @tparam MetricT The metric search space.
//    @param  tree    The vantage point tree containing the item database.
//    @param  head    The start of the item index range.
//    @param  tail    The end of the item index range.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Leaf::Leaf(VpTree* tree,
                            const IndexT& head,
                            const IndexT& tail)
   : Node(tree)
   , theHead(head)
   , theTail(tail)
{
}

}

#include "VpTree_Leaf.hpp"

#endif
