//*****************************************************************************
// FILE:        VpTree_Node.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for vantage-point tree nodes.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************
#ifndef Damkjer_VpTree_Node_HEADER
#define Damkjer_VpTree_Node_HEADER

#include "VpTree.h"  // ISA Damkjer::VpTree inner class;

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree::Node
//>   An abstract node interface for vantage point tree elements.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
class VpTree<MetricT>::Node
{
public:
   Node();
      //> Instantiate a null node.
      //<

   Node(VpTree* const);
      //> Instantiate a node with a database reference.
      //<
    
   virtual ~Node();
      //> Destruct the node and deallocate resources.
      //<
    
   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const = 0;
      //> Accumulate the k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const = 0;
      //> Accumulate the fixed-radius nearest neighbor results.
      //<

protected:
   VpTree<MetricT>* theTree;
      //> The tree that this node belongs to.
      //
      //  Establishes an explicit relationship between the containing class and
      //  nested class.  Included to avoid reallocating contained element data
      //  for each node. As a side effect, nodes organize the tree's data
      //  contents through the construction process.
      //<
};

//*****************************************************************************
// VpTree::Node::Node
//>   Instantiate a null node.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Node::Node()
   : theTree(0)
{
}

//*****************************************************************************
// VpTree::Node::Node
//>   Instantiate a node with a database reference.
//
//    @tparam MetricT The metric search space.
//    @param  tree    The vantage point tree containing the item database.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Node::Node(VpTree* tree)
   : theTree(tree)
{
}

//*****************************************************************************
// VpTree::Node::~Node
//>   Destruct the node and deallocate resources. Virtual to ensure proper node
//    destruction.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::Node::~Node()
{
   theTree = 0;
}

}

#endif
