//*****************************************************************************
// FILE:        VpTree_ResultsCandidate.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for vantage-point tree search result candidates.
//<
//
// LIMITATIONS:
//>   This class template file is a section of the Damkjer::VpTree interface
//    definition and should not be directly included.
//<
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************
#ifndef Damkjer_VpTree_ResultsCandidate_HEADER
#define Damkjer_VpTree_ResultsCandidate_HEADER

#if _OPENMP
#include <omp.h>
#endif

#include <algorithm>
#include <iostream>

#include "VpTree.h"

namespace Damkjer
{

//*****************************************************************************
// VpTree::ResultsCandidate
//>   A light-weight representation of a search result candidate item.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename MetricT>
class VpTree<MetricT>::ResultsCandidate
{
public:

   //***
   // Use the compiler-generated default constructor.
   // ResultsCandidate();
   //***

   ResultsCandidate(const IndexT&, const DistT&);
      //> Instantiate a light-weight results candidate.
      //<
    
   //***
   // Use the compiler-generated copy constructor.
   // ResultsCandidate(const ResultsCandidate&);
   //***

   ~ResultsCandidate(){}
      //> The default destructor. Intentionally non-virtual since
      //  ResultsCandidate is a private inner class on VpTree.
      //
      //  @tparam MetricT The metric search space.
      //<

   //***
   // Use the compiler-generated assignment operator.
   // ResultsCandidate& operator=(const ResultsCandidate&);
   //***

   bool operator<(const ResultsCandidate&) const;
      //> Compare result candidate distances to determine which is closer
      //  to the query.
      //<

private:
   IndexT theIndex;
      //> The index of the candidate in the internal database.
      //<
    
   DistT theDistance;
      //> The candidate's distance to the query point.
      //<

   friend class VpTree<MetricT>;
      //> Provide access to results candidate internals to the containing tree
      //  class.
      //<
};

//*****************************************************************************
// VpTree::ResultsCandidate::ResultsCandidate
//>   Instantiate a light-weight results candidate.
//
//    @tparam MetricT  The metric search space.
//    @param  index    The candidate database index.
//    @param  distance The candidate distance to the query.
//<
//*****************************************************************************
template<typename MetricT>
inline
VpTree<MetricT>::ResultsCandidate::ResultsCandidate(const IndexT& index,
                                                    const DistT& distance)
   : theIndex(index)
   , theDistance(distance)
{
}

//*****************************************************************************
// VpTree::ResultsCandidate::operator<
//>   Less-than comparison for vantage point tree results candidates.
//
//    @note
//    In general, the other members of the comparison operator family should be
//    implemented. They are not in the case of vantage point tree results
//    candidates because the less-than comparitor is only used in sorting
//    semantics. The remaining operators would thus be defined, but never used
//    since items are private to the vantage point tree.
//
//    @param other The result candidate for comparison against this item.
//    @return      true if and only if this candidate is closer to the query
//                 than the other candidate.
//<
//*****************************************************************************
template<typename MetricT>
inline
bool
VpTree<MetricT>::ResultsCandidate::operator<(const ResultsCandidate& other)
const
{
   return theDistance < other.theDistance;
}

}

#endif
