//*****************************************************************************
// FILE:        TestVpTree.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: TestVpTree
//
//    See TestVpTree header for detailed description.
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iomanip>
#include <sstream>
#include <limits>
#include <cmath>
#include <vector>
#include <typeinfo>

#include "TestVpTree.h"
#include "Util/Exception/Exception.h"
#include "Util/SpatialIndexing/Metrics/Metrics.h"
#include "Util/SpatialIndexing/VpTree/VpTree.h"
#include "Util/Streams/BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: testEuclideanKnn
//*****************************************************************************
template <typename PointT>
class testEuclideanKnn : public TestCase::TestStep
{
public:
   testEuclideanKnn()
      : TestCase::TestStep((std::string(
                            "k nearest-neighbor algorithm in Euclidean space "
                            "for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testEuclideanKnn::operator(TestCase&)");

      typedef typename PointT::value_type   ValueT;
      typedef EuclideanDistance<PointT>     MetricT;
      typedef typename MetricT::return_type DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the 20 nearest neighbors to point 45.
      SearchResultsT results = database.knn(data[45], 20);

      //***
      // In Euclidean space, the results should belong to the following set:
      //
      //-    24 25 26
      //- 33 34 35 36 37
      //- 43 44 45 46 47
      //- 53 54 55 56 57
      //-    64 65 66
      //
      // With distances:
      //
      //-         2.23607 2.00000 2.23607
      //- 2.23607 1.41421 1.00000 1.41421 2.23607
      //- 2.00000 1.00000 0.00000 1.00000 2.00000
      //- 2.23607 1.41421 1.00000 1.41421 2.23607
      //-         2.23607 2.00000 2.23607
      //
      // However, the set consists of 21 members, so one item will be missing.
      // Only the furthest points are candidates to be missing:
      //
      //- 24, 26, 33, 37, 53, 57, 64, 66
      //
      //***

      // Display the results
      test.report() << "Find 20 nearest neighbors to point 45: (4,5)...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 20 neighbors... ";
      check = (idxs.size() == 20);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {        25,
                           34, 35, 36,
                       43, 44, 45, 46, 47,
                           54, 55, 56,   
                               65};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 13 out of 13 near neighbors are found
         test.report() << "Check that all 13 of 13 nearest neighbors are "
                       << "found... ";
         check = (present == 13);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check furthest elements
      {
         IndexT f[] = {    24,     26,
                       33,             37,
                      
                       53,             57,
                           64,     66};

         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 7 out of 8 furthest neighbors are found
         test.report() << "Check that only 7 of 8 furthest neighbors are "
                       << "found... ";
         check = (present == 7);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// FUNCTOR: testEuclideanRnn
//*****************************************************************************
template <typename PointT>
class testEuclideanRnn : public TestCase::TestStep
{
public:
   testEuclideanRnn()
      : TestCase::TestStep((std::string(
                            "radially nearest-neighbor algorithm in Euclidean "
                            "space for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testEuclideanRnn::operator(TestCase&)");

      typedef typename PointT::value_type ValueT;
      typedef EuclideanDistance<PointT>   MetricT;
      typedef typename MetricT::return_type        DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the nearest neighbors to point 45 within 3 units.
      ValueT radius = 3;
      SearchResultsT results = database.rnn(data[45], radius);

      //***
      // In Euclidean space, the results should belong to the following set:
      //
      //-          15
      //-    23 24 25 26 27
      //-    33 34 35 36 37
      //- 42 43 44 45 46 47 48
      //-    53 54 55 56 57
      //-    63 64 65 66 67
      //-          75
      //
      // With distances:
      //
      //-                         3.00000
      //-         2.82843 2.23607 2.00000 2.23607 2.82843
      //-         2.23607 1.41421 1.00000 1.41421 2.23607
      //- 3.00000 2.00000 1.00000 0.00000 1.00000 2.00000 3.00000
      //-         2.23607 1.41421 1.00000 1.41421 2.23607
      //-         2.82843 2.23607 2.00000 2.23607 2.82843
      //-                         3.00000
      //
      //***

      // Display the results
      test.report() << "Find nearest neighbors to point 45 (4,5) within 3 "
                    << "units...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 29 neighbors... ";
      check = (idxs.size() == 29);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {            15,
                           23, 24, 25, 26, 27,
                           33, 34, 35, 36, 37,
                       42, 43, 44, 45, 46, 47, 48,
                           53, 54, 55, 56, 57,
                           63, 64, 65, 66, 67,
                                   75};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 13 out of 13 near neighbors are found
         test.report() << "Check that all 29 nearest neighbors are "
                       << "found... ";
         check = (present == 29);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check distances
      {
         check = true;
         // Check that no distance exceeds the search reach
         test.report() << "Check that no distance exceeds the search "
                       << "reach... ";

         for (std::size_t dist = dists.size(); dist --> 0;)
         {
            check &= (dists[dist] <= radius);
         }

         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// FUNCTOR: testManhattanKnn
//*****************************************************************************
template <typename PointT>
class testManhattanKnn : public TestCase::TestStep
{
public:
   testManhattanKnn()
      : TestCase::TestStep((std::string(
                            "k nearest-neighbor algorithm in Manhattan space "
                            "for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testManhattanKnn::operator(TestCase&)");

      typedef typename PointT::value_type ValueT;
      typedef ManhattanDistance<PointT>   MetricT;
      typedef typename MetricT::return_type        DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the 20 nearest neighbors to point 45.
      SearchResultsT results = database.knn(data[45], 20);

      //***
      // In Manhattan space, the results should belong to the following set:
      //
      //-          15   
      //-       24 25 26  
      //-    33 34 35 36 37 
      //- 42 43 44 45 46 47 48
      //-    53 54 55 56 57 
      //-       64 65 66  
      //-          75   
      //
      // With distances:
      //
      //-           3
      //-        3  2  3
      //-     3  2  1  2  3
      //-  3  2  1  0  1  2  3
      //-     3  2  1  2  3
      //-        3  2  3
      //-           3
      //
      // However, the set consists of 25 members, so five items will be
      // missing. Only the furthest points are candidates to be missing:
      //
      //- 15, 24, 26, 33, 37, 42, 48, 53, 57, 64, 66, 75
      //
      //***

      // Display the results
      test.report() << "Find 20 nearest neighbors to point 45: (4,5)...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 20 neighbors... ";
      check = (idxs.size() == 20);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {        25,
                           34, 35, 36,
                       43, 44, 45, 46, 47,
                           54, 55, 56,   
                               65};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 13 out of 13 near neighbors are found
         test.report() << "Check that all 13 of 13 nearest neighbors are "
                       << "found... ";
         check = (present == 13);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check furthest elements
      {
         IndexT f[] = {            15,
                               24,     26,
                           33,             37,
                       42,                     48,
                           53,             57,
                               64,     66,
                                   75};

         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 7 out of 8 furthest neighbors are found
         test.report() << "Check that only 7 of 8 furthest neighbors are "
                       << "found... ";
         check = (present == 7);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// FUNCTOR: testManhattanRnn
//*****************************************************************************
template <typename PointT>
class testManhattanRnn : public TestCase::TestStep
{
public:
   testManhattanRnn()
      : TestCase::TestStep((std::string(
                            "radially nearest-neighbor algorithm in Manhattan "
                            "space for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testManhattanRnn::operator(TestCase&)");

      typedef typename PointT::value_type ValueT;
      typedef ManhattanDistance<PointT>   MetricT;
      typedef typename MetricT::return_type        DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the nearest neighbors to point 45 within 3 units.
      ValueT radius = 3;
      SearchResultsT results = database.rnn(data[45], radius);

      //***
      // In Manhattan space, the results should belong to the following set:
      //
      //-          15
      //-       24 25 26
      //-    33 34 35 36 37
      //- 42 43 44 45 46 47 48
      //-    53 54 55 56 57
      //-       64 65 66
      //-          75
      //
      // With distances:
      //
      //-           3
      //-        3  2  3
      //-     3  2  1  2  3
      //-  3  2  1  0  1  2  3
      //-     3  2  1  2  3
      //-        3  2  3
      //-           3
      //
      //***

      // Display the results
      test.report() << "Find nearest neighbors to point 45 (4,5) within 3 "
                    << "units...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 25 neighbors... ";
      check = (idxs.size() == 25);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {            15,
                               24, 25, 26,
                           33, 34, 35, 36, 37,
                       42, 43, 44, 45, 46, 47, 48,
                           53, 54, 55, 56, 57,
                               64, 65, 66,
                                   75};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 13 out of 13 near neighbors are found
         test.report() << "Check that all 25 nearest neighbors are "
                       << "found... ";
         check = (present == 25);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check distances
      {
         check = true;
         // Check that no distance exceeds the search reach
         test.report() << "Check that no distance exceeds the search "
                       << "reach... ";

         for (std::size_t dist = dists.size(); dist --> 0;)
         {
            check &= (dists[dist] <= radius);
         }

         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// FUNCTOR: testChebyshevKnn
//*****************************************************************************
template <typename PointT>
class testChebyshevKnn : public TestCase::TestStep
{
public:
   testChebyshevKnn()
      : TestCase::TestStep((std::string(
                            "k nearest-neighbor algorithm in Chebyshev space "
                            "for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testManhattanKnn::operator(TestCase&)");

      typedef typename PointT::value_type ValueT;
      typedef ChebyshevDistance<PointT>   MetricT;
      typedef typename MetricT::return_type        DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the 20 nearest neighbors to point 45.
      SearchResultsT results = database.knn(data[45], 20);

      //***
      // In Chebyshev space, the results should belong to the following set:
      //
      //- 23 24 25 26 27
      //- 33 34 35 36 37
      //- 43 44 45 46 47
      //- 53 54 55 56 57 
      //- 63 64 65 66 67
      //
      // With distances:
      //
      //-  2  2  2  2  2
      //-  2  1  1  1  2
      //-  2  1  0  1  2
      //-  2  1  1  1  2
      //-  2  2  2  2  2
      //
      // However, the set consists of 25 members, so five items will be
      // missing. Only the furthest points are candidates to be missing:
      //
      //- 23, 24, 25, 26, 27, 33, 37, 43, 47, 53, 57, 63, 64, 65, 66, 67
      //
      //***

      // Display the results
      test.report() << "Find 20 nearest neighbors to point 45: (4,5)...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 20 neighbors... ";
      check = (idxs.size() == 20);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {34, 35, 36,
                       44, 45, 46,
                       54, 55, 56};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 9 out of 9 near neighbors are found
         test.report() << "Check that all 9 of 9 nearest neighbors are "
                       << "found... ";
         check = (present == 9);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check furthest elements
      {
         IndexT f[] = {23, 24, 25, 26, 27,
                       33,             37,
                       43,             47,
                       53,             57,
                       63, 64, 65, 66, 67};

         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 11 out of 16 furthest neighbors are found
         test.report() << "Check that only 11 of 16 furthest neighbors are "
                       << "found... ";
         check = (present == 11);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// FUNCTOR: testChebyshevRnn
//*****************************************************************************
template <typename PointT>
class testChebyshevRnn : public TestCase::TestStep
{
public:
   testChebyshevRnn()
      : TestCase::TestStep((std::string(
                            "radially nearest-neighbor algorithm in Chebyshev "
                            "space for points of type: ") + 
                           typeid(PointT).name()))
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testChebyshevRnn::operator(TestCase&)");

      typedef typename PointT::value_type ValueT;
      typedef ChebyshevDistance<PointT>   MetricT;
      typedef typename MetricT::return_type        DistT;

      //***
      // Set up a 2D point layout on a regular grid as follows:
      //
      //- (0,0) (0,1) (0,2) ... (0,9)
      //- (1,0) (1,1) (1,2) ... (1,9)
      //-   .     .     .   .     .
      //-   .     .     .    .    .
      //-   .     .     .     .   .
      //- (9,0) (9,1) (9,2) ... (9,9)
      //
      // Make sure points are created in top-left to bottom-right row-major
      // order to force known indices:
      //
      //-  0  1  2 ...  9
      //- 10 11 12 ... 19
      //-  .  .  . .    .
      //-  .  .  .  .   .
      //-  .  .  .   .  .
      //- 90 91 92 ... 99
      //
      //***

      std::deque<PointT> data;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<ValueT>(row));
            pt.push_back(static_cast<ValueT>(col));
            data.push_back(pt);
         }
      }

      // Create the spatially indexed database
      typedef VpTree<MetricT> VpTreeT;

      VpTreeT database(data);
      typedef typename VpTreeT::IndexT         IndexT;
      typedef typename VpTreeT::SearchResultsT SearchResultsT;

      // Perform a search for the nearest neighbors to point 45 within 3 units.
      ValueT radius = 3;
      SearchResultsT results = database.rnn(data[45], radius);

      //***
      // In Chebyshev space, the results should belong to the following set:
      //
      //- 12 13 14 15 16 17 18
      //- 22 23 24 25 26 27 28
      //- 32 33 34 35 36 37 38
      //- 42 43 44 45 46 47 48
      //- 52 53 54 55 56 57 58
      //- 62 63 64 65 66 67 68
      //- 72 73 74 75 76 77 78
      //
      // With distances:
      //
      //-  3  3  3  3  3  3  3
      //-  3  2  2  2  2  2  3
      //-  3  2  1  1  1  2  3
      //-  3  2  1  0  1  2  3
      //-  3  2  1  1  1  2  3
      //-  3  2  2  2  2  2  3
      //-  3  3  3  3  3  3  3
      //
      //***

      // Display the results
      test.report() << "Find nearest neighbors to point 45 (4,5) within 3 "
                    << "units...\n";
      std::deque<IndexT> idxs = results.first;
      std::deque<DistT>          dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (std::size_t idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";

      // Check the results
      bool passedAll = true;
      bool check;

      test.report() << "Check that index and distance set sizes match... ";
      check = (idxs.size() == dists.size());
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      test.report() << "Check that we found 49 neighbors... ";
      check = (idxs.size() == 49);
      passedAll &= check;
      test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";

      // Check nearest elements
      {
         IndexT f[] = {12, 13, 14, 15, 16, 17, 18,
                       22, 23, 24, 25, 26, 27, 28,
                       32, 33, 34, 35, 36, 37, 38,
                       42, 43, 44, 45, 46, 47, 48,
                       52, 53, 54, 55, 56, 57, 58,
                       62, 63, 64, 65, 66, 67, 68,
                       72, 73, 74, 75, 76, 77, 78};
         std::vector<IndexT> furthest(f,
                                        f + sizeof(f)/sizeof(IndexT));

         unsigned char present = 0;
         typename std::deque<IndexT>::iterator pos;

         for (std::size_t nbr = furthest.size(); nbr --> 0;)
         {
            pos = find(idxs.begin(), idxs.end(), furthest[nbr]);

            if (pos != idxs.end()) ++present;
         }

         // Check that 49 out of 49 near neighbors are found
         test.report() << "Check that all 49 nearest neighbors are "
                       << "found... ";
         check = (present == 49);
         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      // Check distances
      {
         check = true;
         // Check that no distance exceeds the search reach
         test.report() << "Check that no distance exceeds the search "
                       << "reach... ";

         for (std::size_t dist = dists.size(); dist --> 0;)
         {
            check &= (dists[dist] <= radius);
         }

         passedAll &= check;
         test.report() << ((check) ? "[OK]" : "[FAILED]") << "\n";
      }

      return passedAll;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// CONSTRUCTOR: TestVpTree::TestVpTree(int, char**)
//*****************************************************************************
TestVpTree::TestVpTree(int argc, char** argv)
   : TestCase(argc, argv,
              "VpTree: Vantage Point Tree Test Suite",
              "VpTree_01",
              "This test case tests the vantage point tree data structure.")
{
   EXCEPTION_TRY("Damkjer::TestVpTree::TestVpTree(int, char**)");

   typedef std::vector<float>  FltPtT;
   typedef std::vector<double> DblPtT;

   registerStep(new testEuclideanKnn<FltPtT>());
   registerStep(new testEuclideanKnn<DblPtT>());
   registerStep(new testEuclideanRnn<FltPtT>());
   registerStep(new testEuclideanRnn<DblPtT>());
   registerStep(new testManhattanKnn<FltPtT>());
   registerStep(new testManhattanKnn<DblPtT>());
   registerStep(new testManhattanRnn<FltPtT>());
   registerStep(new testManhattanRnn<DblPtT>());
   registerStep(new testChebyshevKnn<FltPtT>());
   registerStep(new testChebyshevKnn<DblPtT>());
   registerStep(new testChebyshevRnn<FltPtT>());
   registerStep(new testChebyshevRnn<DblPtT>());

   // Need to add radial tests.

   EXCEPTION_RETHROW;
}

}
