//*****************************************************************************
// FILE:        BlockIndent.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   This class performs the function of block-indenting a stream. It is a
//    convenience manipulator based heavily on the code provided at:
//
//    http://stackoverflow.com/questions/5280858/indenting-paragraph-with-cout
//<
//
// LIMITATIONS: 
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-JUL-26  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <ostream>

#include "BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// BlockIndent static variable instantiation
//*****************************************************************************
const int       BlockIndent::soleIndentationIndex = std::ios_base::xalloc();
   //> The singular index into the stream buffer extensible iword array for the
   //  current indentation level.
   //<   

std::deque<int> BlockIndent::soleIndentationRegistry;
   //> The singular indentation registry used to track block indentation
   //  levels.
   //<   

//*****************************************************************************
// BlockIndent::operator()(std::ostream&)
//>   Manipulate the given output stream with a block indentation style. This
//    method provides a functor interface for operating on std::ostream
//    objects. This function cannot be templatized, otherwise compilers will
//    inline the method, thus exposing the private static variables on the
//    class to the shared library interface. Further, this method is only valid
//    for std::ostream objects since it manipulates the object's
//    std::ios_base::iword database in an output streaming context.
//
//    @param output The output stream to manipulate.
//    @return       The manipulated output stream.
//<
//*****************************************************************************
Util_API std::ostream& BlockIndent::operator()(std::ostream& output) const
{
   // Blocks should always start on a new line, but we don't always need a new
   // line to start a block. What to do?
   output.iword(soleIndentationIndex) += theNumSpaces;
   soleIndentationRegistry.push_front(theNumSpaces);

   return output;
}

//*****************************************************************************
// endBlock(std::ostream&)
//>   End the current block indentation in the specified output stream.
//
//    @param output The output stream to modify.
//    @return       The modified output stream.
//<
//*****************************************************************************
Util_API std::ostream& endBlock(std::ostream& output)
{
   // Blocks should always end with a new line, but we don't always need a new
   // line to end a block. What to do?
//   output << "\n";

   if (!BlockIndent::soleIndentationRegistry.empty())
   {
      int decr = BlockIndent::soleIndentationRegistry.front();
      BlockIndent::soleIndentationRegistry.pop_front();
      output.iword(BlockIndent::soleIndentationIndex) -= decr;
   }
   
   return output;
}

}
