//*****************************************************************************
// FILE:        FormatFilter.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//> The class implementation for the output format filter.
//<
//
// LIMITATIONS:
//> No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-AUG-04  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iostream>

#include "Util/Exception/Exception.h"

#include "BlockIndent.h"
#include "FormatFilter.h"

namespace Damkjer
{
//*****************************************************************************
// FormatFilter::soleLineWidthIndex
//> The index for the line width in the stream buffer extensible iword array.
//<
//*****************************************************************************
const int
FormatFilter::soleLineWidthIndex = std::ios_base::xalloc();

//*****************************************************************************
// FormatFilter::FormatFilter(std::ostream&, std::size_t)
//>   Instantiate a format filter.
//
//    @param output    The stream to wrap for formatting.
//    @param lineWidth The word-wrap line length.
//<
//*****************************************************************************
Util_API
FormatFilter::FormatFilter(std::ostream& output, std::size_t lineWidth)
   : theCount(0)
   , theDestination(output)
   , theDestinationBuf(output.rdbuf())
   , theLineWidth(lineWidth)
{
   EXCEPTION_TRY("Damkjer::FormatFilter::FormatFilter(std::ostream&, "
                 "std::size_t)");

   theDestination.iword(widthIndex()) = static_cast<long>(lineWidth);
   theDestination.rdbuf(this);

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// FormatFilter::~FormatFilter
//>   Destruct the format filter and deallocate resources.
//<
//*****************************************************************************
Util_API
FormatFilter::~FormatFilter()
{
   if (!theBuffer.empty())
   {
      // flush the buffer, if needed.
      overflow('\0');
   }

   theDestination.rdbuf(theDestinationBuf);
}

//*****************************************************************************
// FormatFilter::overflow
//>   Format the stream with line-wrapping and block indentation.
//
//    The overflow method is called automatically when streambuf internal
//    buffers become full and flush their contents. This method basically
//    implements a line-buffering stream buffer.
//
//    The algorithm is:
//
//    - Explicit end of line ("\r" or "\n"): we flush our buffer to the
//      underlying stream's buffer, and set our record of the line length to 0.
//    - An "alert" character: sent to the underlying stream without recording
//      its length, since it doesn't normally affect the a appearance of the
//      output.
//    - Tab: treated as moving to the next tab stop, which is assumed as
//      happening every tab_width characters.
//    - Everything else: really basic buffering with word wrapping. We try to
//      add the character to the buffer, and if it exceeds our line width, we
//      search for the last space/tab in the buffer and break the line there.
//      If there is no space/tab, we break the line at the limit.
//
//    @param c The character to be placed into the controlled output stream.
//    @return  The put character.
//<
//*****************************************************************************
std::streambuf::int_type
FormatFilter::overflow(int_type c)
{
   EXCEPTION_TRY("Damkjer::FormatFilter::overflow(int_type)");

   if (traits_type::eq_int_type(traits_type::eof(), c))
   {
      return traits_type::not_eof(c);
   }

   std::streamsize bufferSize = static_cast<std::streamsize>(theBuffer.size());
   std::streamsize blockIndent = theDestination.iword(BlockIndent::index());
   std::size_t indentAsSize = static_cast<std::size_t>(blockIndent);
   std::size_t blockWidth  = (theLineWidth > indentAsSize)
                             ? theLineWidth - indentAsSize
                             : 1;

   const char* prefix = std::string(indentAsSize, ' ').c_str();

   char_type character = static_cast<char_type>(c);

   switch (c)
   {
   case '\0':
      theCount = 0;
      theDestinationBuf->sputn(prefix, blockIndent);
      theDestinationBuf->sputn(theBuffer.c_str(), bufferSize);
      theBuffer.clear();
      return c;

   case '\n':
   case '\r':
      //***
      // Explicit end of line: we flush our buffer to the underlying stream's
      // buffer, and set our record of the line length to 0.
      //***

      theBuffer += character;
      theCount = 0;
      theDestinationBuf->sputn(prefix, blockIndent);
      theDestinationBuf->sputn(theBuffer.c_str(), bufferSize + 1);
      theBuffer.clear();
      return c;

   case '\a':
      //***
      // An "alert" character: sent to the underlying stream without recording
      // its length, since it doesn't normally affect the a appearance of the
      // output.
      //***

      return theDestinationBuf->sputc(character);

   case '\t':
      //***
      // Tab: treated as moving to the next tab stop, which is assumed as
      // happening every TAB_WIDTH characters.
      //***

      theBuffer += character;
      theCount += soleTabWidth - theCount % soleTabWidth;

      return c;

   default:
      //***
      // Everything else: really basic buffering with word wrapping. We try to
      // add the character to the buffer, and if it exceeds our line width, we
      // search for the last space/tab in the buffer and break the line there.
      // If there is no space/tab, we break the line at the limit.
      //***
      if (theCount >= blockWidth)
      {
         std::size_t pos = theBuffer.find_last_of("\\/ \t");
         std::streamsize posAsStrmSz = static_cast<std::streamsize>(pos);

         if (pos != FilterString::npos)
         {
            theDestinationBuf->sputn(prefix, blockIndent);

            switch (theBuffer[pos])
            {
            case '\\':
            case '/':
               theDestinationBuf->sputn(theBuffer.c_str(), posAsStrmSz+1);
               break;
            default:
               theDestinationBuf->sputn(theBuffer.c_str(), posAsStrmSz);
            }

            theCount = theBuffer.size()-pos-1;
            theBuffer = FilterString(theBuffer, pos+1);
         }
         else
         {
            theDestinationBuf->sputn(prefix, blockIndent);
            theDestinationBuf->sputn(theBuffer.c_str(), bufferSize);
            theBuffer.clear();
            theCount = 0;
         }

         theDestinationBuf->sputc('\n');
      }

      theBuffer += character;
      ++theCount;

      return c;
   }

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// sline(std::ostream&)
//>   Insert a line of hyphen ('-') characters into the argument stream
//    creating the appearance of a single horizontal line.
//
//    @param output The stream to be updated.
//    @return       The updated stream.
//<
//*****************************************************************************
Util_API
std::ostream&
sline(std::ostream& output)
{
   EXCEPTION_TRY("Damkjer::sline(std::ostream&)");

   std::size_t lineWidth = static_cast<std::size_t>(
                                     output.iword(FormatFilter::widthIndex()));

   if (lineWidth == 0)
   {
      lineWidth = 79;
      output.iword(FormatFilter::widthIndex()) = 79;
   }

   std::size_t blockIndent = static_cast<std::size_t>(
                                           output.iword(BlockIndent::index()));
   std::size_t blockWidth  = lineWidth - blockIndent;

   std::string line(blockWidth, '-');

   output << line.c_str() << "\n";
   
   return output;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// dline(std::ostream&)
//>   Insert a line of equals ('=') characters into the argument stream
//    creating the appearance of a double horizontal line.
//
//    @param output The stream to be updated.
//    @return       The updated stream.
//<
//*****************************************************************************
Util_API
std::ostream&
dline(std::ostream& output)
{
   EXCEPTION_TRY("Damkjer::dline(std::ostream&)");

   std::size_t lineWidth = static_cast<std::size_t>(
                                     output.iword(FormatFilter::widthIndex()));

   if (lineWidth == 0)
   {
      lineWidth = 79;
      output.iword(FormatFilter::widthIndex()) = 79;
   }

   std::size_t blockIndent = static_cast<std::size_t>(
                                           output.iword(BlockIndent::index()));
   std::size_t blockWidth  = lineWidth - blockIndent;

   std::string line(blockWidth, '=');

   output << line.c_str() << "\n";
   
   return output;

   EXCEPTION_RETHROW;
}

}
