//*****************************************************************************
// FILE:        ManipulatorBase.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   Base class interface definition for stream manipulators.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2013-AUG-03  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef Damkjer_ManipulatorBase_HEADER
#define Damkjer_ManipulatorBase_HEADER

namespace Damkjer
{

//*****************************************************************************
// CLASS: ManipulatorBase
//>   A template base class for stream manipulators. This class is modeled on
//    the "Curiously Recurring Template Pattern". This pattern provides a
//    simple interface for stream manipulators to be called with anonymous
//    objects on input and output streams. It is an example of "static
//    polymorphism".
//
//    @tparam ManipulatorT The stream manipulator type.
//<
//*****************************************************************************
template <class ManipulatorT>
class ManipulatorBase
{
public:
   template <class StreamT>
   StreamT& operator()(StreamT& stream) const
   {
      // call Manipulator::operator()
      static_cast<const ManipulatorT&>(*this)(stream);
      return stream;
   }
      //> Invoke the ManipulatorT functor on the stream. The function operator
      //  allows manipulators derived from ManipulatorBase to behave as
      //  functions with state.
      //
      //  @tparam StreamT The stream type.
      //  @param  stream  The stream to be manipulated.
      //  @return         The manipulated stream.
      //<
};

}

//*****************************************************************************
// FUNCTION: operator<<(OstreamT&, const ManipulatorBase<ManipulatorT>&)
//>   Manipulate an output stream with a manipulator via stream insertion.
//
//    @tparam OstreamT     The output stream type.
//    @tparam ManipulatorT The output stream manipulator type.
//    @param  outputStream The output stream to be manipulated.
//    @param  manipulate   The output stream manipulator functor.
//    @return              The manipulated output stream.
//<
//*****************************************************************************
template <class OstreamT, class ManipulatorT>
inline OstreamT&
operator<<(OstreamT& outputStream,
           const Damkjer::ManipulatorBase<ManipulatorT>& manipulate)
{
   return manipulate(outputStream);
}

//*****************************************************************************
// FUNCTION: operator>>(IstreamT&, const ManipulatorBase<ManipulatorT>&)
//>   Manipulate an input stream with a manipulator via stream extraction.
//
//    @tparam IstreamT     The input stream type.
//    @tparam ManipulatorT The input stream manipulator type.
//    @param  inputStream  The input stream to be manipulated.
//    @param  manipulate   The input stream manipulator functor.
//    @return              The manipulated input stream.
//<
//*****************************************************************************
template <class IstreamT, class ManipulatorT>
inline IstreamT&
operator>>(IstreamT& inputStream,
           const Damkjer::ManipulatorBase<ManipulatorT>& manipulate)
{
   return manipulate(inputStream);
}

#endif
