//*****************************************************************************
// FILE:        TestCase.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   The class implementation for unit test cases.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <sstream>
#include <iomanip>
#include <ctime>

#include "Util/Exception/Exception.h"
#include "Util/Streams/BlockIndent.h"

#include "TestCase.h"

namespace Damkjer
{

//*****************************************************************************
// TestCase::TestCase(int, char**, const std::string&, const std::string&,
//                    const std::string&, std::ostream&)
//>   Intialize a test case with required information.
//
//    This constructor sets the name, ID, and description for the unit test
//    case. It also wraps an output stream with a report stream to easily
//    redirect and format test logs.
//
//    @param argc            The command-line argument count.
//    @param argv            The command-line arguments.
//    @param caseName        The test case descriptive name.
//    @param caseID          The test case identifier.
//    @param caseDescription The test case verbose description.
//    @param reportStream    The test case report stream. Defaults to standard
//                           log output.
//<
//*****************************************************************************
Util_API
TestCase::TestCase(int                argc,
                   char**             argv,
                   const std::string& caseName,
                   const std::string& caseID,
                   const std::string& caseDescription,
                   std::ostream&      reportStream)
   : theStartTime()
   , theCaseName(caseName)
   , theCaseID(caseID)
   , theCaseDescription(caseDescription)
   , theReportStream(reportStream)
   , theStreamFormatter(reportStream)
   , theArgv(argv)
   , theDetailsIndent(0)
   , theArgc(argc)
   , theStepCount(0)
{
}
   
//*****************************************************************************
// TestCase::argv(int)
//>   The requested command line argument, converted to a string.
//
//    @param  index      the index of the desired command line argument.
//    @return            the requested command line argument as a std::string.
//    @throw  OutOfRange when the reqested argument is outside the range of
//                       available arguments.
//<
//*****************************************************************************
Util_API
std::string TestCase::argv(int index) const
{
   EXCEPTION_TRY("Damkjer::TestCase::argv(int)");

   if (index >= 0 && index < theArgc)
   {
      return theArgv[index];
   }

   //***
   // In released code, there's not a whole lot that the user can do about
   // this problem. This is a logic error and should be prevented through
   // appropriate bounds checking on the input prior to calling the method.
   //***
   std::ostringstream msg;

   msg << "Error encountered while retrieving an argument vector (argv) "
       << "element.\n\n"
       << "Unable to access the requested element (" << index << ") because "
       << "it is outside the valid bounds of [0, " << theArgc << ").";

   throw OutOfRange(msg.str(),  MODULE, __FILE__, __LINE__);

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::execute()
//>   Execute the test case ad generate formatted test report. This method
//    performs the simple sequence:
//       -# Output Header to the report stream
//       -# Run steps for the test
//       -# Output Footer to the report stream
//
//    @return EXIT_SUCCESS on successful execution of all test steps.
//            EXIT_FAILURE otherwise.
//<
//*****************************************************************************
Util_API
int
TestCase::execute()
{
   EXCEPTION_TRY("Damkjer::TestCase::execute()");

   header();

   theReportStream << BlockIndent(4);
   // Indent the body of the test report by 4.
   int results = steps();
   theReportStream << endBlock;

   footer();

   return results;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::testing(const std::string&)
//>   The formatted test leader.
//
//    Return a string in the following format for consistent reports:
//-    "<step_number>. Testing <description>..."
//
//    @param  description The description to include in the test leader.
//    @return             The formatted testing string.
//<
//*****************************************************************************
void
TestCase::testing(const std::string& description)
{
   std::ostringstream oss;
   oss << ++theStepCount << ". ";
   
   theDetailsIndent = static_cast<long>(oss.str().length());

   theReportStream << oss.str() << "Testing " << description << "...\n";

   return;
}

//*****************************************************************************
// TestCase::header()
//>   Header for the unit test case report entry. This method is also
//    responsible for starting the test case timer.
//
//    @return The formatted test report header.
//<
//*****************************************************************************
std::ostream&
TestCase::header()
{
#if ( _WIN32 || _WIN64 )
   QueryPerformanceCounter(&theStartTime);
#elif ( __linux || __unix || __posix )
   clock_gettime(CLOCK_MONOTONIC, &theStartTime);
#else
   // How else can we capture high-resolution timing information?
#endif

   std::time_t now = std::time(NULL);

   theReportStream
      << dline
      << " TEST CASE   :  " << theCaseID << "\n"
      << " NAME        :  " << theCaseName << "\n"
      << "\n"
      << " DESCRIPTION :\n";

   // Indent this block by 4.
   theReportStream << BlockIndent(4);
   theReportStream << theCaseDescription << "\n";
   theReportStream << endBlock;

   theReportStream
      << dline
      << sline
      << " START TIME  :  " << std::asctime(std::localtime(&now))
      << sline;

   return theReportStream;
}

//*****************************************************************************
// TestCase::steps()
//>   Execute the registered steps for the test case.
//
//    @return EXIT_SUCCESS on successful execution of all test steps.
//            EXIT_FAILURE otherwise.
//<
//*****************************************************************************
int
TestCase::steps()
{
   EXCEPTION_TRY("Damkjer::TestCase::steps()");

   bool testPassed = false;
   unsigned int passedTests = 0;

   for (unsigned int i = 0; i < theSteps.size(); ++i)
   {
      testing(theSteps[i]->description());
      theReportStream << BlockIndent(theDetailsIndent);
      testPassed = (*(theSteps[i]))(*this);
      theReportStream << "\n" << ((testPassed) ? "[PASSED]" : "[FAILED]")
                      << "\n" << endBlock;

      if (testPassed) ++passedTests;

      theReportStream << sline;
   }
   
   theReportStream << "Summary: " << passedTests << "/" << theSteps.size()
                   << " tests passed.\n";

   return (passedTests == theSteps.size()) ? EXIT_SUCCESS : EXIT_FAILURE;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::footer
//>   Footer for the unit test case report entry. This method is also
//    responsible for stopping the test case timer.
//
//    @return The formatted test report footer.
//<
//*****************************************************************************
std::ostream&
TestCase::footer()
{
   double elapsed = 0;

#if ( _WIN32 || _WIN64 )
   TimeT stopTime;
   QueryPerformanceCounter(&stopTime);

   TimeT frequency;
   QueryPerformanceFrequency(&frequency);

   elapsed = 1.0 * (stopTime.QuadPart - theStartTime.QuadPart) /
             frequency.QuadPart;
#elif ( __linux || __unix || __posix )
   TimeT stopTime;
   clock_gettime(CLOCK_MONOTONIC, &stopTime);

   elapsed = (stopTime.tv_sec - theStartTime.tv_sec) +
             (stopTime.tv_nsec - theStartTime.tv_nsec) * 1e-9;
#endif

   unsigned int iElapsed = static_cast<unsigned int>(elapsed);

   std::stringstream msg;
   
   char cfill = msg.fill('0');

   msg << std::setw(2) << iElapsed / 3600 << ":"
       << std::setw(2) << ((iElapsed / 60) % 60) << ":"
       << std::setw(9) << std::fixed << std::setprecision(6)
       << (elapsed-iElapsed+(iElapsed % 60));
   
   msg.fill(cfill);

   std::time_t now = std::time(NULL);

   theReportStream
      << sline
      << " STOP TIME   :  " << std::asctime(localtime(&now))
      << sline
      << dline
      << " ELAPSED     :  " << msg.str() << "\n"
      << dline;

   return theReportStream;
}

}
