//*****************************************************************************
// FILE:        TestCase.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for unit test cases.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef Damkjer_TestCase_HEADER
#define Damkjer_TestCase_HEADER

#if ( _WIN32 || _WIN64 )
   #if NOMINMAX      // Avoid macro definitions of min/max infavor of STL
#include <windows.h> // USES LARGE_INTEGER union
   #else
      #define NOMINMAX // Avoid macro definitions of min/max in favor of STL
#include <windows.h> // USES LARGE_INTEGER union
      #undef NOMINMAX
   #endif
#elif ( __linux || __unix || __posix )
#include <time.h> // USES timespec structure
#else
   // How else can we capture high-resolution timing information?
#endif

#include <iostream> // HASA std::ostream interface and data member.
#include <string>   // HASA std::string data member.
#include <vector>   // HASA std::vector data member.

// ISA component of the Util library API.
#include "Util/UtilAPI.h"

// HASA FormatFilter data member.
#include "Util/Streams/FormatFilter.h"

namespace Damkjer
{

//*****************************************************************************
// CLASS: TestCase
//>   A common base class for unit test cases.
//<
//*****************************************************************************
class TestCase
{
public:
#if ( _WIN32 || _WIN64 )
   typedef LARGE_INTEGER TimeT;
      //> The time structure type for Windows systems.
      //<
#elif ( __linux || __unix || __posix )
   typedef timespec TimeT;
      //> The time structure type for *nix systems.
      //<
#else
   // How else can we capture high-resolution timing information?
#endif

   class TestStep;

   Util_API TestCase(int argc, char**,
                     const std::string&,
                     const std::string&,
                     const std::string&,
                     std::ostream& reportStream=std::clog);
      //> Intialize a test case with required information.
      //<
   
   Util_API virtual ~TestCase(){}
      //> Destruct the TestCase and deallocate resources.
      //<

   Util_API virtual int execute();
      //> Execute the test case ad generate formatted test report.
      //<

   Util_API virtual std::ostream& report();
      //> The test case report output stream.
      //<

   Util_API std::size_t registerStep(TestStep* const);
      //> Register a test step with the test case for execution.
      //<

   Util_API int stepNum() const { return theStepCount; }
      //> Identify the current step in test exection.
      //
      //  @return The current step in execution.
      //<

   Util_API int argc() const { return theArgc; }
      //> The command line argument count.
      //
      //  @return The command line argument count.
      //<
   
   Util_API std::string argv(int) const;
      //> The requested command line argument, converted to a string.
      //<

private:
   TestCase();
      //> Explicitly disable the compiler-generated default constructor. No
      //  definition provided.
      //<

   TestCase(const TestCase&);
      //> Explicitly disable the compiler-generated copy constructor. No
      //  definition provided.
      //<

   TestCase& operator=(const TestCase&);
      //> Explicitly disable the compiler-generated assignment operator. No
      //  definition provided.
      //
      //  @return The updated left-hand side when not disabled.
      //<

   void testing(const std::string&);
      //> The formatted test leader.
      //<

   std::ostream& header();
      //> Header for the unit test case report entry.
      //<
   
   int steps();
      //> Execute the registered steps for the test case.
      //<

   std::ostream& footer();
      //> Footer for the unit test case report entry.
      //<

   TimeT theStartTime;
      //> The test case start time.
      //<

   std::string theCaseName;
      //> The test case name.
      //<

   std::string theCaseID;
      //> The test case identifier.
      //<

   std::string theCaseDescription;
      //> The test case description.
      //<
   
   std::ostream& theReportStream;
      //> The report output stream.
      //<

   std::vector<TestStep*> theSteps;
      //> The test steps for evaluation.
      //<

   FormatFilter theStreamFormatter;
      //> The test report formatter.
      //<

   char** theArgv;
      //> The vector of arguments.
      //<

   long theDetailsIndent;
      //> The current report indentation level.
      //<

   int theArgc;
      //> The argument count.
      //<

   int theStepCount;
      //> The test step counter.
      //<
};

//*****************************************************************************
// CLASS: TestCase::TestStep
//>   An individual step of a comprehensive test case.
//<
//*****************************************************************************
class TestCase::TestStep
{
public:
   Util_API explicit TestStep(const std::string& description)
      : theDescription(description)
   {
   }
      //> Instantiate a TestStep with a description.
      //
      //  @param description The test step description.
      //<

   Util_API virtual ~TestStep(){}
      //> Destruct the TestStep and deallocate resources.
      //<

   Util_API const std::string& description() const { return theDescription; }
      //> The description of the test step.
      //
      //  @return The test step description.
      //<

   Util_API virtual bool operator()(TestCase&) const = 0;
      //> Perform the evaluation of the test step.
      //
      //  @return true if and only if the test step is successful.
      //<

private:
   TestStep();
      //> Explicitly disable the compiler-generated default constructor. No
      //  definition provided.
      //<

   TestStep(const TestStep&);
      //> Explicitly disable the compiler-generated copy constructor. No
      //  definition provided.
      //<

   TestStep& operator=(const TestStep&);
      //> Explicitly disable the compiler-generated assignment operator. No
      //  definition provided.
      //
      //  @return The updated left-hand side when not disabled.
      //<

   std::string theDescription;
      //> The description of the test step.
      //<
};

//*****************************************************************************
// TestCase::report
//>   The test case report output stream.
//
//    @return the report stream for this test case.
//<
//*****************************************************************************
Util_API
inline
std::ostream&
TestCase::report()
{
   return theReportStream;
}

//*****************************************************************************
// TestCase::registerStep
//>   Register a test step with the test case for execution.
//
//    @param step a pointer to a step to be executed as part of this test case.
//    @return     the test case step number associated with the registered
//                step.
//<
//*****************************************************************************
Util_API
inline
std::size_t
TestCase::registerStep(TestStep* const step)
{
   theSteps.push_back(step);
   return theSteps.size();
}


}

#endif
