% timebar   Create a time-based waitbar.
%
% File: timebar.m
%
% Description:
%    This MATLAB function creates a waitbar dialog with an estimated time to
%    completion based on linear extrapolation from number of processing steps
%    completed, number of processing steps remaining and current elapsed time.
%
% Limitations:
%    None.
%
% Synopsis:
%    [handle] = timebar(step, steps, message, start, handle)
%
% Inputs:
%    step     - Current processing step.
%    steps    - Total steps in process.
%    message  - Base message to be included in waitbar text.
%    start    - Process start time.
%    handle   - Figure handle for existing waitbar.
%
% Outputs:
%    handle  - Figure handle for rendered timebar graphic.
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    waitbar
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.

%******************************************************************************
% timebar
%******************************************************************************
function [handle] = timebar(step, steps, message, start, handle)
   % Create a waitbar dialog with an estimated time to completion based on
   % linear extrapolation from number of processing steps completed, number of 
   % processing steps remaining and current elapsed time.
   %
   % Parameters:
   %    step    - Current processing step.
   %
   %    steps   - Total steps in process.
   %
   %    message - Base message to be included in waitbar text.
   %
   %    start   - Process start time.
   %
   %    handle  - Figure handle for existing waitbar.
   %
   % Returns:
   %    handle - Figure handle for rendered timebar graphic.

step    = double(step);
steps   = double(steps);
elapsed = toc(start);
total   = floor(elapsed/step*steps);
elapsed = floor(elapsed);
remain  = total-elapsed;

% Construct total string
tmsg = timeString(total);

% Construct elapsed string
emsg = timeString(elapsed);

% Construct total string
rmsg = timeString(remain);

% Construct waitbar message
wbmsg = [message emsg '/' tmsg ' (' rmsg ' remaining)'];

% Create a new waitbar if the handle is invalid or wasn't passed in. Otherwise
% render to the valid figure handle.
if (nargin > 4 && all(ishghandle(handle, 'figure')))
    waitbar(step/steps, handle, wbmsg);
else
    handle = waitbar(step/steps, wbmsg);
end

end

%******************************************************************************
% timeString
%******************************************************************************
function [message] = timeString(seconds)
   % Create a human-readable,  order of magnitude appropriate interpretation of
   % a time value.
   %
   % Parameters:
   %    seconds - Time as decimal seconds.
   %
   % Returns:
   %    message - Coarse order of magnitude interpretation of the seconds
   %              value.

   if (seconds > 7200)
      message = [num2str(floor(seconds/3600)) 'h'];
   elseif (seconds > 120)
      message = [num2str(floor(seconds/60)) 'm'];
   else
      message = [num2str(seconds) 's'];
   end
% message = sprintf('%02d:%02d:%02d',...
%                   floor(seconds/3600),...
%                   mod(floor(seconds/60),60),...
%                   floor(mod(seconds,60)));
end

