%%
% [CLASSES] = SPANALYZE(QUERIES, DATABASE, ...)
%
% Attribute points by eigen analysis.
%
%   Copyright 2012 Kristian L. Damkjer
%
%   Software History:
%      2013-JAN-28   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Change output to a struct for easier referencing.
%

function [ CLASSES ] = spanalyze( QUERIES, DATABASE, varargin )
% Parse Inputs
userParams = parseInputs(varargin{:});

dimensions=size(QUERIES, 1);
elements=size(QUERIES, 2);

norms=zeros(size(QUERIES));
feats=zeros(size(QUERIES));

biases=zeros(1,elements);

ints=zeros(1,elements);

msg='Computing Structure Tensors...';
tstart=tic;
h = timebar(1, elements, msg, tstart);

step=1000;
tic;
for elem=1:step:elements
   
   last=min(elem+step-1,elements);
   
   % Get the nearest neighbors of elem
   if (userParams.neighbors <= 0)
       % Perform a fixed radius search
       NN=DATABASE.rnn(QUERIES(:,elem:last),...
                       userParams.radius);
   else
       if (userParams.radius <= 0)
           % Search unconstrained
           NN=DATABASE.knn(QUERIES(:,elem:last),...
                           userParams.neighbors);
       else
           % Search constrained to radius
           NN=DATABASE.knn(QUERIES(:,elem:last),...
                           userParams.neighbors,...
                           'lim',userParams.radius);
       end
   end

   [covs,bias,inty]=fastcov(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));

% Same solution via SVD appears to be slower via emperical testing
%    recentered=fastcenter(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));

   for nbr=1:size(NN,1)
      % skip underconstrained (possible with radius searches)
      if (length(NN{nbr})<5)
         continue;
      end

      [V,D]=eig(covs{nbr});
      [~,index]=min(diag(D));
%       norms(:,elem+nbr-1)=abs(V(:,index)) .* sqrt(D(index, index));
      norms(:,elem+nbr-1)=abs(V(:,index));
      feats(:,elem+nbr-1)=sort(diag(D),'descend');

      biases(elem+nbr-1)=bias{nbr};
      ints(elem+nbr-1)=inty{nbr};

% Same solution via SVD appears to be slower via emperical testing
%       [~,S,V]=svd(recentered{nbr},0);
%       S = S ./ sqrt(length(NN{nbr})-1);
%       norms(:,elem+nbr-1)=abs(V(:,dimensions) .* S(dimensions, dimensions));
%       feats(:,elem+nbr-1)=diag(S).^2;
      
   end
   
   if (toc > 1)
      tic;
      h = timebar(elem, elements, msg, tstart, h);
   end
end

if (all(ishghandle(h, 'figure')))
   close(h);
end

% Compute Omnivariance
omnivar=prod(feats,1).^(1/dimensions);

% Compute Eigen Entropy
energy=sum(feats,1);
nfeats=bsxfun(@times,feats',1./energy')';
nfeats(isnan(nfeats))=1;
entropy=-sum(nfeats.*log(nfeats))./log(dimensions);
entropy(entropy==1)=0;

% Compute Eigen Structure
structure=1-entropy;

% Compute Fractional Anisotropy
evmeans=mean(feats,1);
evdevs=feats-repmat(evmeans,dimensions,1);
numer=dimensions.*sum(evdevs.^2,1);
denom=(dimensions-1).*sum(feats.^2,1);
fa=(numer./denom).^(1/2);

% Compute Anisotropy
ani=(feats(1,:)-feats(dimensions,:))./(feats(1,:));

% Compute Isotropy
iso=1-ani;

% Compute dimensional degree
dims=(feats(1:dimensions-1,:)-feats(2:dimensions,:))./...
     repmat(feats(1,:),dimensions-1,1);

% Compute Dimensional Embedding
[~,embed]=max([dims;iso],[],1);

% Populate feature classes
%CLASSES=[feats;norms;dims;iso;ani;fa;entropy;structure;omnivar;embed;biases;ints];
CLASSES.features=feats;
CLASSES.normals=norms;
CLASSES.dimensionality=dims;
CLASSES.isotropy=iso;
CLASSES.anisotropy=ani;
CLASSES.FA=fa;
CLASSES.entropy=entropy;
CLASSES.structure=structure;
CLASSES.omnivariance=omnivar;
CLASSES.embedding=embed;
CLASSES.biases=biases;
CLASSES.intensity=ints;

% st=zeros(size(QUERIES));
% 
% msg='Computing Normal Structure Tensors...';
% tstart=tic;
% h = timebar(1, elements, msg, tstart);
% 
% for elem=1:step:elements
%     
%    last=min(elem+step-1,elements);
%     
%    % Get the nearest neighbors of elem
%    if (userParams.neighbors <= 0)
%        % Perform a fixed radius search
%        NN=DATABASE.rnn(QUERIES(:,elem:last),...
%                        userParams.radius);
%    else
%        if (userParams.radius <= 0)
%            % Search unconstrained
%            NN=DATABASE.knn(QUERIES(:,elem:last),...
%                            userParams.neighbors);
%        else
%            % Search constrained to radius
%            NN=DATABASE.knn(QUERIES(:,elem:last),...
%                            userParams.neighbors,...
%                            'lim',userParams.radius);
%        end
%    end
% 
%    for nbr=1:size(NN,1)
%       normals=norms(:,NN{nbr});
% 
%       S=svd(normals',0);
%       st(:,elem+nbr-1)=S.*S./(length(NN{nbr})-1);
% 
% %       D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %       st(:,elem+nbr-1)=sort(D,'descend');
% 
% %       ST=normals*normals';
% 
% %       if (nbr == 1)
% %           D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %           disp('EIG:');
% %           disp(sqrt(D));
% %           
% %           S=svd(normals',0);
% %           S = S ./ sqrt(length(NN{nbr})-1);
% %           disp('SVD:');
% %           disp(abs(S));
% % 
% %       end
% 
%    end
% 
%    if(toc > 1)
%       tic;
%       h = timebar(elem, elements, msg, tstart, h);
%    end
% end
% 
% if (all(ishghandle(h, 'figure')))
%    close(h);
% end

% CLASSES = QUERIES;

end

%%
% PARSEINPUTS    Support function to parse inputs into userParams structure
function [userParams] = parseInputs(varargin)

userParams = struct('radius', 0, 'neighbors', 0);

if length(varargin) == 1 || ~isnumeric(varargin{2})
    value = varargin{1};

    if (isscalar(value)  && ...
        isreal(value)    && ...
        value >= 5)
       userParams.neighbors = fix(value);
    else
       error('Damkjer:InvalidCount', ...
             ['Number of Neighbors must be a real valued positive '...
              'integer greater or equal to 5: ' num2str(value)]);
    end

    varargin(1) = [];
end

% Parse the Property/Value pairs
if rem(length(varargin), 2) ~= 0
   error('Damkjer:PropertyValueNotPair', ...
         ['Additional arguments must take the form of Property/Value '...
          'pairs']);
end

propertyNames = {'neighbors', 'radius'};

while ~isempty(varargin)
   property = varargin{1};
   value    = varargin{2};

   % If the property has been supplied in a shortened form, lengthen it
   iProperty = find(strncmpi(property, propertyNames, length(property)));

   if isempty(iProperty)
      error('Damkjer:InvalidProperty', 'Invalid Property');
   elseif length(iProperty) > 1
      error('Damkjer:AmbiguousProperty', ...
            'Supplied shortened property name is ambiguous');
   end
   
   property = propertyNames{iProperty};

   switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
            userParams.neighbors = fix(value);
         else
       error('Damkjer:InvalidCount', ...
             ['Number of Neighbors must be a real valued positive '...
              'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidPointWeights', ...
                  'Radius must be a real valued positive scalar');
         end
   end

    varargin(1:2) = [];
end

if (userParams.neighbors <= 0 && userParams.radius <= 0)
   userParams.radius = 1;
end

end

