//*****************************************************************************
// FILE:        TestException.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: TestException
//
//    See TestTestCase header for detailed description.
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <sstream>

#include <vector>
#include <typeinfo>

#include "damkjerConfig.h"
#include "TestException.h"
#include "Util/Streams/BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: throwTest
//*****************************************************************************
template <typename CatchT, typename ThrowT=CatchT>
class throwTest : public TestCase::TestStep
{
public:
   throwTest(const ThrowT& expected)
      : TestCase::TestStep(typeid(CatchT).name())
      , theExpected(expected)
   {
   }
      //> 
      //<
   
   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::throwTest::operator(TestCase&)");

      try
      {
         throw theExpected;

         test.report() << "Did not catch an exception!\n";

         // If we get here, we didn't throw: failed test.
         return false;
      }
      catch (const CatchT& e)
      {
         test.report() << "Caught expected exception:\n";
         test.report() << blockIndent() << e.what() << "\n" << endBlock;

         // If we get here, we threw the correct exception: successful test.
         return true;
      }
      catch (...)
      {
         test.report() << "Caught unexpected exception.\n";

         // If we get here, we threw something unexpected: failed test.
         return false;
      }

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
   ThrowT theExpected;
      //>
      //<
};

//*****************************************************************************
// CONSTRUCTOR: TestException::TestException(int, char**)
//*****************************************************************************
TestException::TestException(int argc, char** argv)
   : TestCase(argc, argv,
              "Exception: Basic Functionality",
              "Exception_01",
              "This test case tests basic Exception functionality.")
{
   EXCEPTION_TRY("Damkjer::TestException::TestException(int, char**)");

   DomainError     dde("Function not defined for provided input. (undefined "
                       "behavior)",
                       MODULE,__FILE__,__LINE__);
   InvalidArgument dia("Argument cannot be interpretted in the context of the "
                       "function. (e.g., bitset initialized with char other "
                       "than '0' or '1')",
                       MODULE,__FILE__,__LINE__);
   LengthError     dle("Action exceeded a maximum allowable size.",
                       MODULE,__FILE__,__LINE__);
   OutOfRange      dor("Parameter outside of permitted range. (e.g., bad "
                       "index)",
                       MODULE,__FILE__,__LINE__);
   RangeError      dre("Error in internal computation. (result cannot be "
                       "represented)",
                       MODULE,__FILE__,__LINE__);
   OverflowError   doe("Detected arithmetic overflow. (result is larger "
                       "than type can represent)",
                       MODULE,__FILE__,__LINE__);
   UnderflowError  due("Detected arithmetic underflow. (result is smaller "
                       "than type can represent)",
                       MODULE,__FILE__,__LINE__);
   std::domain_error     sde("Function not defined for provided input. "
                             "(undefined behavior)");
   std::invalid_argument sia("Argument cannot be interpretted in the context "
                             "of the function. (e.g., bitset initialized with "
                             "char other than '0' or '1')");
   std::length_error     sle("Action exceeded a maximum allowable size.");
   std::out_of_range     sor("Parameter outside of permitted range. (e.g., "
                             "bad index)");
   std::range_error      sre("Error in internal computation. (result cannot "
                             "be represented)");
   std::overflow_error   soe("Detected arithmetic overflow. (result is larger "
                             "than type can represent)");
   std::underflow_error  sue("Detected arithmetic underflow. (result is "
                             "smaller than type can represent)");
   std::bad_alloc        sba("Unable to allocate requested memory.");
   std::bad_cast         sbc("Failure to dynamic cast. Cast relationship is "
                             "defined, but results in an incomplete object of"
                             "the destination type");
   std::bad_exception    sbe("Exception thrown that wasn't listed in the "
                             "function dynamic exception specifier.");
   std::bad_typeid       sbt("typeid operator applied to dereferenced null "
                             "pointer.");

   registerStep(new throwTest<std::domain_error>(sde));
   registerStep(new throwTest<std::invalid_argument>(sia));
   registerStep(new throwTest<std::length_error>(sle));
   registerStep(new throwTest<std::out_of_range>(sor));
   registerStep(new throwTest<std::range_error>(sre));
   registerStep(new throwTest<std::overflow_error>(soe));
   registerStep(new throwTest<std::underflow_error>(sue));
   registerStep(new throwTest<std::bad_alloc>(sba));
   registerStep(new throwTest<std::bad_cast>(sbc));
   registerStep(new throwTest<std::bad_exception>(sbe));
   registerStep(new throwTest<std::bad_typeid>(sbt));
   registerStep(new throwTest<std::logic_error>(sde));
   registerStep(new throwTest<std::logic_error>(sia));
   registerStep(new throwTest<std::logic_error>(sle));
   registerStep(new throwTest<std::logic_error>(sor));
   registerStep(new throwTest<std::runtime_error>(sre));
   registerStep(new throwTest<std::runtime_error>(soe));
   registerStep(new throwTest<std::runtime_error>(sue));
   registerStep(new throwTest<std::exception>(sde));
   registerStep(new throwTest<std::exception>(sia));
   registerStep(new throwTest<std::exception>(sle));
   registerStep(new throwTest<std::exception>(sor));
   registerStep(new throwTest<std::exception>(sre));
   registerStep(new throwTest<std::exception>(soe));
   registerStep(new throwTest<std::exception>(sue));
   registerStep(new throwTest<std::exception>(sba));
   registerStep(new throwTest<std::exception>(sbc));
   registerStep(new throwTest<std::exception>(sbe));
   registerStep(new throwTest<std::exception>(sbt));
   registerStep(new throwTest<DomainError>(dde));
   registerStep(new throwTest<InvalidArgument>(dia));
   registerStep(new throwTest<LengthError>(dle));
   registerStep(new throwTest<OutOfRange>(dor));
   registerStep(new throwTest<RangeError>(dre));
   registerStep(new throwTest<OverflowError>(doe));
   registerStep(new throwTest<UnderflowError>(due));
   registerStep(new throwTest<LogicError, DomainError>(dde));
   registerStep(new throwTest<LogicError, InvalidArgument>(dia));
   registerStep(new throwTest<LogicError, LengthError>(dle));
   registerStep(new throwTest<LogicError, OutOfRange>(dor));
   registerStep(new throwTest<RuntimeError, RangeError>(dre));
   registerStep(new throwTest<RuntimeError, OverflowError>(doe));
   registerStep(new throwTest<RuntimeError, UnderflowError>(due));
   registerStep(new throwTest<Exception, DomainError>(dde));
   registerStep(new throwTest<Exception, InvalidArgument>(dia));
   registerStep(new throwTest<Exception, LengthError>(dle));
   registerStep(new throwTest<Exception, OutOfRange>(dor));
   registerStep(new throwTest<Exception, RangeError>(dre));
   registerStep(new throwTest<Exception, OverflowError>(doe));
   registerStep(new throwTest<Exception, UnderflowError>(due));

   EXCEPTION_RETHROW;
}

}
