//*****************************************************************************
// FILE:        ManhattanDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: ManhattanDistance
//    This class describes a Manhattan distance metric. Distances between two
//    points, p and q, in n-dimensional Manhattan space are calculated as
//    follows:
//
//    d(p,q)=sum{i=0:n}(|p[i] - q[i]|).
//
// LIMITATIONS: ManhattanDistance
//    The ManhattanDistance functor must operate on an STL container class, or
//    a class that supports "duck" typing through a public typedef named
//    "value_type".
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef ManhattanDistance_HEADER
#define ManhattanDistance_HEADER

#include <cmath>
#include <iterator>

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: ManhattanDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = PointT::value_type>
class ManhattanDistance
{
public:
   typedef typename PointT  value_type;
      //> Add support for "duck" typing by publishing the point type for this
      //  functor.
      //<

   typedef typename ReturnT return_type;
      //> Add support for "duck" typing by publishing the return type for this
      //  functor.
      //<
   
   //**************************************************************************
   // ManhattanDistance::operator()
   //**************************************************************************
   ReturnT operator()(const PointT& a, const PointT& b) const
   {
      // For predictable data fidelity, perform computations with the same
      // fidelity as the point-type.
      PointT::value_type sum = 0;
      
      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         sum += std::abs(*ai - *bi);
      }
      
      // Then static-cast to the desired return type.
      return static_cast<ReturnT>(sum);
   }
      //> The operator() function performs the distance calculation for
      //  Metrics functors. Distances between two points, p and q, in
      //  n-dimensional Manhattan space are calculated as follows:
      //
      //    d(p,q)=sum{i=0:n}(|p[i] - q[i]|).
      //<
};

}

#endif