//*****************************************************************************
// FILE:        MinkowskiDistance.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: MinkowskiDistance
//    This class describes a Minkowski distance metric. The Minkowski distance
//    must be defined with a power parameter, p. Distances between two
//    points, a and b, in n-dimensional Minkowski space are calculated as
//    follows:
//
//    d(a,b)=(sum{i=0:n}((a[i] - b[i])^p))^(1/p).
//
// LIMITATIONS: MinkowskiDistance
//    The MinkowskiDistance functor must operate on an STL container class, or
//    a class that supports "duck" typing through a public typedef named
//    "value_type".
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//  2013-AUG-09  K. Damkjer
//               Moved into its own header to improve maintainability.
//<
//*****************************************************************************

#ifndef MinkowskiDistance_HEADER
#define MinkowskiDistance_HEADER

#include <cmath>
#include <iterator>

#include "damkjerConfig.h"

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: MinkowskiDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = PointT::value_type>
class MinkowskiDistance
{
public:
   typedef typename PointT  value_type;
      //> Add support for "duck" typing by publishing the point type for this
      //  functor.
      //<

   typedef typename ReturnT return_type;
      //> Add support for "duck" typing by publishing the return type for this
      //  functor.
      //<
   
   //**************************************************************************
   // MinkowskiDistance::MinkowskiDistance
   //**************************************************************************
   MinkowskiDistance()
      : theP(static_cast<PointT::value_type>(2))
   {
   }

   //**************************************************************************
   // MinkowskiDistance::MinkowskiDistance
   //**************************************************************************
   explicit MinkowskiDistance(typename PointT::value_type p)
      : theP(static_cast<PointT::value_type>(p))
   {
      EXCEPTION_TRY("Damkjer::MinkowskiDistance<PointT, ReturnT>::"
                    "MinkowskiDistance");

      if (theP < 1)
      {
         //***
         // In released code, there's not a whole lot that the user can do
         // about this problem. This is a logic error and should be prevented
         // through appropriate bounds checking on the input prior to
         // calling the constructor.
         //***
         std::ostringstream msg;

         msg << "Error encountered while defining a Minkowski distance "
             << "metric.\n\n"
             << "Unable to define a Minkowski distance metric with the "
             << "requested p-value: " << theP << ". The Minkowski distance "
             << "must be defined with p >= 1 to describe a metric space.";

         throw DomainError(msg.str(),  MODULE, __FILE__, __LINE__);
      }

      EXCEPTION_RETHROW;
   }

   //**************************************************************************
   // MinkowskiDistance::operator()
   //**************************************************************************
   ReturnT operator()(const PointT& a, const PointT& b) const
   {
      // For predictable data fidelity, perform computations with the same
      // fidelity as the point-type.
      PointT::value_type sum = 0;
   
      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         sum += std::pow(std::abs(*ai - *bi), theP);
      }
   
      // Then static-cast to the desired return type.
      static const typename PointT::value_type ONE =
         static_cast<PointT::value_type>(1.);
      return static_cast<ReturnT>(std::pow(sum, ONE/theP));
   }
      //> The operator() function performs the distance calculation for
      //  Metrics functors. Distances between two points, p and q, in
      //  n-dimensional Minkowski space are calculated as follows:
      //
      //    d(a,b)=(sum{i=0:n}((a[i] - b[i])^p))^(1/p).
      //<

protected:
   typename PointT::value_type theP;
};

}

#endif