//*****************************************************************************
// FILE:        TestMetrics.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: TestMetrics
//
//    See TestMetrics header for detailed description.
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iomanip>
#include <sstream>
#include <limits>
#include <cmath>
#include <vector>
#include <typeinfo>

#include "damkjerConfig.h"
#include "TestMetrics.h"
#include "Util/SpatialIndexing/Metrics/Metrics.h"
#include "Util/Streams/BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// FUNCTION: approxEqual
//*****************************************************************************
template<typename T>
bool approx_equal(T a, T b, double absTol = std::numeric_limits<T>::epsilon())
{
   T inMax = (std::abs(a) > std::abs(b))
             ? std::abs(a)
             : std::abs(b);

   inMax = (inMax > static_cast<T>(1.0))
           ? inMax
           : static_cast<T>(1.0);

   return (std::abs(a - b) <= absTol * inMax);
}

//*****************************************************************************
// FUNCTOR: testEuclidean
//*****************************************************************************
template <typename PointT, typename ReturnT>
class testEuclidean : public TestCase::TestStep
{
public:
   testEuclidean()
      : TestCase::TestStep(typeid(theMetric).name())
      , theMetric()
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testEuclidean::operator(TestCase&)");

      PointT a;
      a.push_back(static_cast<PointT::value_type>(0.3));
      a.push_back(static_cast<PointT::value_type>(0.5));

      PointT b;
      b.push_back(static_cast<PointT::value_type>(0.7));
      b.push_back(static_cast<PointT::value_type>(0.8));

      ReturnT fromMetric = theMetric(a, b);

      //***
      // Expected value is sqrt((0.7-0.3)*(0.7-0.3)+(0.8-0.5)*(0.8-0.5))
      //                   sqrt(0.4*0.4+0.3*0.3)
      //                   sqrt(0.16+0.09)
      //                   sqrt(0.25)
      //                   0.5
      //***
      bool passed =
         approx_equal(fromMetric, static_cast<ReturnT>(0.5),
                      std::numeric_limits<PointT::value_type>::epsilon());

      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
                    << "Expected result            : "
                    << std::setprecision(12) << static_cast<ReturnT>(0.5) << "\n";

      return passed;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
   EuclideanDistance<PointT,ReturnT> theMetric;
      //>
      //<
};

//*****************************************************************************
// FUNCTOR: testManhattan
//*****************************************************************************
template <typename PointT, typename ReturnT>
class testManhattan : public TestCase::TestStep
{
public:
   testManhattan()
      : TestCase::TestStep(typeid(theMetric).name())
      , theMetric()
   {
   }
      //>
      //<

   bool approx_equal(ReturnT a, ReturnT b) const
   {
      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
      double absTolO = std::numeric_limits<ReturnT>::epsilon();

      double absTol = (absTolI > absTolO) ? absTolI : absTolO;

      ReturnT inMax = (std::abs(a) > std::abs(b))
                      ? std::abs(a)
                      : std::abs(b);

      inMax = (inMax > static_cast<ReturnT>(1.0))
              ? inMax
              : static_cast<ReturnT>(1.0);

      return (std::abs(a - b) <= absTol * inMax);
   }

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testManhattan::operator(TestCase&)");

      PointT a;
      a.push_back(static_cast<PointT::value_type>(0.3));
      a.push_back(static_cast<PointT::value_type>(0.5));

      PointT b;
      b.push_back(static_cast<PointT::value_type>(0.7));
      b.push_back(static_cast<PointT::value_type>(0.8));

      ReturnT fromMetric = theMetric(a, b);

      //***
      // Expected value is |0.7-0.3|+|0.8-0.5|
      //                   0.4+0.3
      //                   0.7
      //***
      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.7));

      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
                    << "Expected result            : "
                    << std::setprecision(12) << static_cast<ReturnT>(0.7) << "\n";

      return passed;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
   ManhattanDistance<PointT,ReturnT> theMetric;
      //>
      //<
};

//*****************************************************************************
// FUNCTOR: testChebyshev
//*****************************************************************************
template <typename PointT, typename ReturnT>
class testChebyshev : public TestCase::TestStep
{
public:
   testChebyshev()
      : TestCase::TestStep(typeid(theMetric).name())
      , theMetric()
   {
   }
      //>
      //<

   bool approx_equal(ReturnT a, ReturnT b) const
   {
      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
      double absTolO = std::numeric_limits<ReturnT>::epsilon();

      double absTol = (absTolI > absTolO) ? absTolI : absTolO;

      ReturnT inMax = (std::abs(a) > std::abs(b))
                      ? std::abs(a)
                      : std::abs(b);

      inMax = (inMax > static_cast<ReturnT>(1.0))
              ? inMax
              : static_cast<ReturnT>(1.0);

      return (std::abs(a - b) <= absTol * inMax);
   }

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testChebyshev::operator(TestCase&)");

      PointT a;
      a.push_back(static_cast<PointT::value_type>(0.3));
      a.push_back(static_cast<PointT::value_type>(0.5));

      PointT b;
      b.push_back(static_cast<PointT::value_type>(0.7));
      b.push_back(static_cast<PointT::value_type>(0.8));

      ReturnT fromMetric = theMetric(a, b);

      //***
      // Expected value is max(|0.7-0.3|,|0.8-0.5|)
      //                   max(0.4,0.3)
      //                   0.4
      //***
      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.4));

      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
                    << "Expected result            : "
                    << std::setprecision(12) << static_cast<ReturnT>(0.4) << "\n";

      return passed;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
   ChebyshevDistance<PointT,ReturnT> theMetric;
      //>
      //<
};

//*****************************************************************************
// FUNCTOR: testMinkowski
//*****************************************************************************
template <typename PointT, typename ReturnT>
class testMinkowski : public TestCase::TestStep
{
public:
   testMinkowski()
      : TestCase::TestStep(typeid(theMetric).name())
      , theMetric(2)
   {
   }
      //>
      //<

   bool approx_equal(ReturnT a, ReturnT b) const
   {
      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
      double absTolO = std::numeric_limits<ReturnT>::epsilon();

      double absTol = (absTolI > absTolO) ? absTolI : absTolO;

      ReturnT inMax = (std::abs(a) > std::abs(b))
                      ? std::abs(a)
                      : std::abs(b);

      inMax = (inMax > static_cast<ReturnT>(1.0))
              ? inMax
              : static_cast<ReturnT>(1.0);

      return (std::abs(a - b) <= absTol * inMax);
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testMinkowski::operator(TestCase&)");

      PointT a;
      a.push_back(static_cast<PointT::value_type>(0.3));
      a.push_back(static_cast<PointT::value_type>(0.5));

      PointT b;
      b.push_back(static_cast<PointT::value_type>(0.7));
      b.push_back(static_cast<PointT::value_type>(0.8));

      ReturnT fromMetric = theMetric(a, b);

      //***
      // Expected value is sqrt((0.7-0.3)*(0.7-0.3)+(0.8-0.5)*(0.8-0.5))
      //                   sqrt(0.4*0.4+0.3*0.3)
      //                   sqrt(0.16+0.09)
      //                   sqrt(0.25)
      //                   0.5
      //***
      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.5));

      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
                    << "Expected result            : "
                    << std::setprecision(12) << static_cast<ReturnT>(0.5) << "\n";

      return passed;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
   MinkowskiDistance<PointT,ReturnT> theMetric;
      //>
      //<
};

//*****************************************************************************
// FUNCTOR: minkowskiThrow
//*****************************************************************************
template <typename PointT, typename ReturnT>
class minkowskiThrow : public TestCase::TestStep
{
public:
   minkowskiThrow()
      : TestCase::TestStep(
                 std::string("throw from ")+
                 std::string(typeid(MinkowskiDistance<PointT,ReturnT>).name()))
   {
   }
      //> This functor constructs with a simple default constructor since the
      //  test body is not performing a direct comparison against an expected
      //  value.
      //<
   
   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::minkowskiThrow::operator(TestCase&)");

      try
      {
         MinkowskiDistance<PointT,ReturnT> metric(0.5); // <-- Throw here.
         test.report() << "Did not catch an exception!\n";

         // If we get here, we didn't throw: failed test.
         return false;
      }
      catch (const DomainError& e)
      {
         test.report() << "Caught expected exception:\n";

         test.report() << blockIndent() << e;
         e.stackTrace(test.report());
         test.report() << endBlock;

         // If we get here, we threw the correct exception: successful test.
         return true;
      }
      catch (...)
      {
         test.report() << "Caught unexpected exception.\n";

         // If we get here, we threw something unexpected: failed test.
         return false;
      }

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<
};

//*****************************************************************************
// CONSTRUCTOR: TestMetrics::TestMetrics(int, char**)
//*****************************************************************************
TestMetrics::TestMetrics(int argc, char** argv)
   : TestCase(argc, argv,
              "Metrics: Metrics Functor Suite",
              "Metrics_01",
              "This test case tests defined distance Metrics.")
{
   EXCEPTION_TRY("Damkjer::TestMetrics::TestMetrics(int, char**)");

   typedef std::vector<float>  FloatPtT;
   typedef std::vector<double> DoublePtT;

   registerStep(new testEuclidean<FloatPtT,float>());
   registerStep(new testEuclidean<FloatPtT,double>());
   registerStep(new testEuclidean<DoublePtT,float>());
   registerStep(new testEuclidean<DoublePtT,double>());
   registerStep(new testManhattan<FloatPtT,float>());
   registerStep(new testManhattan<FloatPtT,double>());
   registerStep(new testManhattan<DoublePtT,float>());
   registerStep(new testManhattan<DoublePtT,double>());
   registerStep(new testChebyshev<FloatPtT,float>());
   registerStep(new testChebyshev<FloatPtT,double>());
   registerStep(new testChebyshev<DoublePtT,float>());
   registerStep(new testChebyshev<DoublePtT,double>());
   registerStep(new testMinkowski<FloatPtT,float>());
   registerStep(new testMinkowski<FloatPtT,double>());
   registerStep(new testMinkowski<DoublePtT,float>());
   registerStep(new testMinkowski<DoublePtT,double>());
   registerStep(new minkowskiThrow<FloatPtT,float>());
   registerStep(new minkowskiThrow<FloatPtT,double>());
   registerStep(new minkowskiThrow<DoublePtT,float>());
   registerStep(new minkowskiThrow<DoublePtT,double>());

   EXCEPTION_RETHROW;
}

}
