//*****************************************************************************
// FILE:        TestVpTree.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: TestVpTree
//
//    See TestVpTree header for detailed description.
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iomanip>
#include <sstream>
#include <limits>
#include <cmath>
#include <vector>
#include <typeinfo>

#include "damkjerConfig.h"
#include "TestVpTree.h"
#include "Util/SpatialIndexing/Metrics/Metrics.h"
#include "Util/SpatialIndexing/VpTree/VpTree.h"
#include "Util/Streams/BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// FUNCTOR: testEuclidean
//*****************************************************************************
template <typename MetricT>
class testKnn : public TestCase::TestStep
{
public:
   testKnn()
      : TestCase::TestStep(typeid(VpTree<MetricT>).name())
   {
   }
      //>
      //<

   bool operator()(TestCase& test) const
   {
      EXCEPTION_TRY("Damkjer::testEuclidean::operator(TestCase&)");

      typedef typename MetricT::value_type PointT;

      std::deque<PointT> theData;
      
      for (unsigned int row = 0; row < 10; ++row)
      {
         for (unsigned int col = 0; col < 10; ++col)
         {
            PointT pt;
            pt.push_back(static_cast<PointT::value_type>(row));
            pt.push_back(static_cast<PointT::value_type>(col));
            theData.push_back(pt);
         }
      }

      typedef VpTree<MetricT> VpTreeT;

      VpTreeT theDatabase(theData);
      VpTreeT::SearchResultsT results = theDatabase.knn(theData[45], 20);

      std::deque<std::size_t> idxs = results.first;
      std::deque<MetricT::return_type> dists = results.second;

      test.report() << "idxs = {";

      if (!idxs.empty())
      {
         test.report() << idxs[0];
         for (unsigned int idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << idxs[idx];
         }
      }

      test.report() << "}\n";

      test.report() << "dists = {";

      if (!idxs.empty())
      {
         test.report() << dists[0];
         for (unsigned int idx = 1; idx < idxs.size(); ++idx)
         {
            test.report() << ", " << dists[idx];
         }
      }

      test.report() << "}\n";
      return true;

      EXCEPTION_RETHROW;
   }
      //> The parentheses operator represents the body of the test step
      //  function. It will perform the necessary evaluation and return true if
      //  and only if the test passes.
      //<

private:
};

////*****************************************************************************
//// FUNCTOR: testManhattan
////*****************************************************************************
//template <typename PointT, typename ReturnT>
//class testManhattan : public TestCase::TestStep
//{
//public:
//   testManhattan()
//      : TestCase::TestStep(typeid(theMetric).name())
//      , theMetric()
//   {
//   }
//      //>
//      //<
//
//   bool approx_equal(ReturnT a, ReturnT b) const
//   {
//      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
//      double absTolO = std::numeric_limits<ReturnT>::epsilon();
//
//      double absTol = (absTolI > absTolO) ? absTolI : absTolO;
//
//      ReturnT inMax = (std::abs(a) > std::abs(b))
//                      ? std::abs(a)
//                      : std::abs(b);
//
//      inMax = (inMax > static_cast<ReturnT>(1.0))
//              ? inMax
//              : static_cast<ReturnT>(1.0);
//
//      return (std::abs(a - b) <= absTol * inMax);
//   }
//
//   bool operator()(TestCase& test) const
//   {
//      EXCEPTION_TRY("Damkjer::testManhattan::operator(TestCase&)");
//
//      PointT a;
//      a.push_back(static_cast<PointT::value_type>(0.3));
//      a.push_back(static_cast<PointT::value_type>(0.5));
//
//      PointT b;
//      b.push_back(static_cast<PointT::value_type>(0.7));
//      b.push_back(static_cast<PointT::value_type>(0.8));
//
//      ReturnT fromMetric = theMetric(a, b);
//
//      //***
//      // Expected value is |0.7-0.3|+|0.8-0.5|
//      //                   0.4+0.3
//      //                   0.7
//      //***
//      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.7));
//
//      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
//                    << "Expected result            : "
//                    << std::setprecision(12) << static_cast<ReturnT>(0.7) << "\n";
//
//      return passed;
//
//      EXCEPTION_RETHROW;
//   }
//      //> The parentheses operator represents the body of the test step
//      //  function. It will perform the necessary evaluation and return true if
//      //  and only if the test passes.
//      //<
//
//private:
//   ManhattanDistance<PointT,ReturnT> theMetric;
//      //>
//      //<
//};
//
////*****************************************************************************
//// FUNCTOR: testChebyshev
////*****************************************************************************
//template <typename PointT, typename ReturnT>
//class testChebyshev : public TestCase::TestStep
//{
//public:
//   testChebyshev()
//      : TestCase::TestStep(typeid(theMetric).name())
//      , theMetric()
//   {
//   }
//      //>
//      //<
//
//   bool approx_equal(ReturnT a, ReturnT b) const
//   {
//      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
//      double absTolO = std::numeric_limits<ReturnT>::epsilon();
//
//      double absTol = (absTolI > absTolO) ? absTolI : absTolO;
//
//      ReturnT inMax = (std::abs(a) > std::abs(b))
//                      ? std::abs(a)
//                      : std::abs(b);
//
//      inMax = (inMax > static_cast<ReturnT>(1.0))
//              ? inMax
//              : static_cast<ReturnT>(1.0);
//
//      return (std::abs(a - b) <= absTol * inMax);
//   }
//
//   bool operator()(TestCase& test) const
//   {
//      EXCEPTION_TRY("Damkjer::testChebyshev::operator(TestCase&)");
//
//      PointT a;
//      a.push_back(static_cast<PointT::value_type>(0.3));
//      a.push_back(static_cast<PointT::value_type>(0.5));
//
//      PointT b;
//      b.push_back(static_cast<PointT::value_type>(0.7));
//      b.push_back(static_cast<PointT::value_type>(0.8));
//
//      ReturnT fromMetric = theMetric(a, b);
//
//      //***
//      // Expected value is max(|0.7-0.3|,|0.8-0.5|)
//      //                   max(0.4,0.3)
//      //                   0.4
//      //***
//      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.4));
//
//      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
//                    << "Expected result            : "
//                    << std::setprecision(12) << static_cast<ReturnT>(0.4) << "\n";
//
//      return passed;
//
//      EXCEPTION_RETHROW;
//   }
//      //> The parentheses operator represents the body of the test step
//      //  function. It will perform the necessary evaluation and return true if
//      //  and only if the test passes.
//      //<
//
//private:
//   ChebyshevDistance<PointT,ReturnT> theMetric;
//      //>
//      //<
//};
//
////*****************************************************************************
//// FUNCTOR: testMinkowski
////*****************************************************************************
//template <typename PointT, typename ReturnT>
//class testMinkowski : public TestCase::TestStep
//{
//public:
//   testMinkowski()
//      : TestCase::TestStep(typeid(theMetric).name())
//      , theMetric(2)
//   {
//   }
//      //>
//      //<
//
//   bool approx_equal(ReturnT a, ReturnT b) const
//   {
//      double absTolI = std::numeric_limits<PointT::value_type>::epsilon();
//      double absTolO = std::numeric_limits<ReturnT>::epsilon();
//
//      double absTol = (absTolI > absTolO) ? absTolI : absTolO;
//
//      ReturnT inMax = (std::abs(a) > std::abs(b))
//                      ? std::abs(a)
//                      : std::abs(b);
//
//      inMax = (inMax > static_cast<ReturnT>(1.0))
//              ? inMax
//              : static_cast<ReturnT>(1.0);
//
//      return (std::abs(a - b) <= absTol * inMax);
//   }
//
//   bool operator()(TestCase& test) const
//   {
//      EXCEPTION_TRY("Damkjer::testMinkowski::operator(TestCase&)");
//
//      PointT a;
//      a.push_back(static_cast<PointT::value_type>(0.3));
//      a.push_back(static_cast<PointT::value_type>(0.5));
//
//      PointT b;
//      b.push_back(static_cast<PointT::value_type>(0.7));
//      b.push_back(static_cast<PointT::value_type>(0.8));
//
//      ReturnT fromMetric = theMetric(a, b);
//
//      //***
//      // Expected value is sqrt((0.7-0.3)*(0.7-0.3)+(0.8-0.5)*(0.8-0.5))
//      //                   sqrt(0.4*0.4+0.3*0.3)
//      //                   sqrt(0.16+0.09)
//      //                   sqrt(0.25)
//      //                   0.5
//      //***
//      bool passed = approx_equal(fromMetric, static_cast<ReturnT>(0.5));
//
//      test.report() << "Value from Euclidean Metric: " << std::setprecision(12) << fromMetric << "\n"
//                    << "Expected result            : "
//                    << std::setprecision(12) << static_cast<ReturnT>(0.5) << "\n";
//
//      return passed;
//
//      EXCEPTION_RETHROW;
//   }
//      //> The parentheses operator represents the body of the test step
//      //  function. It will perform the necessary evaluation and return true if
//      //  and only if the test passes.
//      //<
//
//private:
//   MinkowskiDistance<PointT,ReturnT> theMetric;
//      //>
//      //<
//};
//
//*****************************************************************************
// CONSTRUCTOR: TestVpTree::TestVpTree(int, char**)
//*****************************************************************************
TestVpTree::TestVpTree(int argc, char** argv)
   : TestCase(argc, argv,
              "VpTree: Vantage Point Tree Test Suite",
              "VpTree_01",
              "This test case tests the vantage point tree data structure.")
{
   EXCEPTION_TRY("Damkjer::TestVpTree::TestVpTree(int, char**)");

   typedef std::vector<float>  FltPtT;
   typedef std::vector<double> DblPtT;

   typedef EuclideanDistance<FltPtT> EuclidFltT;
   typedef EuclideanDistance<DblPtT> EuclidDblT;
   typedef ManhattanDistance<FltPtT> ManhattanFltT;
   typedef ManhattanDistance<DblPtT> ManhattanDblT;
   typedef ChebyshevDistance<FltPtT> ChebyshevFltT;
   typedef ChebyshevDistance<DblPtT> ChebyshevDblT;
   typedef MinkowskiDistance<DblPtT> MinkowskiDblT;

   registerStep(new testKnn<EuclidFltT>());
   registerStep(new testKnn<EuclidDblT>());
   registerStep(new testKnn<ManhattanFltT>());
   registerStep(new testKnn<ManhattanDblT>());
   registerStep(new testKnn<ChebyshevFltT>());
   registerStep(new testKnn<ChebyshevDblT>());

   EXCEPTION_RETHROW;
}

}
