//*****************************************************************************
// FILE:        FormatFilter.cpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-AUG-04  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include "damkjerConfig.h"
#include "FormatFilter.h"
#include "BlockIndent.h"

#include <iostream>

namespace Damkjer
{
//*****************************************************************************
// FormatFilter static variable instantiation
//*****************************************************************************
const int FormatFilter::LINE_WIDTH_INDEX = std::ios_base::xalloc();

//*****************************************************************************
// FormatFilter::FormatFilter
//*****************************************************************************
FormatFilter::FormatFilter(std::ostream& output, long lineWidth)
   : theDestination(output)
   , theDestinationBuf(output.rdbuf())
   , theCount(0)
   , theLineWidth(lineWidth)
{
   EXCEPTION_TRY("Damkjer::FormatFilter::FormatFilter(std::ostream&, long)");

   theDestination.iword(widthIndex()) = lineWidth;
   theDestination.rdbuf(this);

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// FormatFilter::~FormatFilter
//*****************************************************************************
Util_API FormatFilter::~FormatFilter()
{
   if (!theBuffer.empty())
   {
      // flush the buffer, if needed.
      overflow(NULL);
   }

   theDestination.rdbuf(theDestinationBuf);
}

//*****************************************************************************
// FormatFilter::overflow
//*****************************************************************************
Util_API std::streambuf::int_type FormatFilter::overflow(int_type c)
{
   EXCEPTION_TRY("Damkjer::FormatFilter::overflow(int_type)");

//   long lineWidth   = theDestination.iword(widthIndex());
   long blockIndent = theDestination.iword(blockIndent::index());
   long blockWidth  = theLineWidth - blockIndent;

   const char* prefix = std::string(blockIndent, ' ').c_str();

   if (traits_type::eq_int_type(traits_type::eof(), c))
   {
      return traits_type::not_eof(c);
   }

   switch (c)
   {
   case NULL:
      theCount = 0;
      theDestinationBuf->sputn(prefix, blockIndent);
      theDestinationBuf->sputn(theBuffer.c_str(), theBuffer.size());
      theBuffer.clear();
      return c;

   case '\n':
   case '\r':
      //***
      // Explicit end of line: we flush our buffer to the underlying stream's
      // buffer, and set our record of the line length to 0.
      //***
      
      theBuffer += static_cast<char_type>(c);
      theCount = 0;
      theDestinationBuf->sputn(prefix, blockIndent);
      theDestinationBuf->sputn(theBuffer.c_str(), theBuffer.size());
      theBuffer.clear();
      return c;

   case '\a':
      //***
      // An "alert" character: sent to the underlying stream without recording
      // its length, since it doesn't normally affect the a appearance of the
      // output.
      //***

      return theDestinationBuf->sputc(static_cast<char_type>(c));

   case '\t':
      //***
      // Tab: treated as moving to the next tab stop, which is assumed as
      // happening every TAB_WIDTH characters.
      //***

      theBuffer += static_cast<char_type>(c);
      theCount += TAB_WIDTH - theCount % TAB_WIDTH;

      return c;

   default:
      //***
      // Everything else: really basic buffering with word wrapping. We try to
      // add the character to the buffer, and if it exceeds our line width, we
      // search for the last space/tab in the buffer and break the line there.
      // If there is no space/tab, we break the line at the limit.
      //***
      if (theCount >= blockWidth)
      {
         size_t wpos = theBuffer.find_last_of("\\/ \t");

         if (wpos != string::npos)
         {
            theDestinationBuf->sputn(prefix, blockIndent);

            switch (theBuffer[wpos])
            {
            case '\\':
            case '/':
               theDestinationBuf->sputn(theBuffer.c_str(), wpos+1);
               break;
            default:
               theDestinationBuf->sputn(theBuffer.c_str(), wpos);
            }

            theCount = theBuffer.size()-wpos-1;
            theBuffer = string(theBuffer, wpos+1);
         }
         else
         {
            theDestinationBuf->sputn(prefix, blockIndent);
            theDestinationBuf->sputn(theBuffer.c_str(), theBuffer.size());
            theBuffer.clear();
            theCount = 0;
         }

         theDestinationBuf->sputc('\n');
      }

      theBuffer += static_cast<char_type>(c);
      ++theCount;

      return c;
   }

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// sline
//*****************************************************************************
Util_API std::ostream& sline(std::ostream& output)
{
   EXCEPTION_TRY("Damkjer::sline(std::ostream&)");

   long lineWidth   = output.iword(FormatFilter::widthIndex());

   if (lineWidth == 0)
   {
      lineWidth = 79;
      output.iword(FormatFilter::widthIndex()) = 79;
   }

   long blockIndent = output.iword(blockIndent::index());
   long blockWidth  = lineWidth - blockIndent;
//   long blockWidth  = (40 > lineWidth - blockIndent) ? 40 : lineWidth - blockIndent;

   std::string line(blockWidth, '-');

   output << line.c_str() << "\n";
   
   return output;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// dline
//*****************************************************************************
Util_API std::ostream& dline(std::ostream& output)
{
   EXCEPTION_TRY("Damkjer::dline(std::ostream&)");

   long lineWidth   = output.iword(FormatFilter::widthIndex());

   if (lineWidth == 0)
   {
      lineWidth = 79;
      output.iword(FormatFilter::widthIndex()) = 79;
   }

   long blockIndent = output.iword(blockIndent::index());
   long blockWidth  = lineWidth - blockIndent;
//   long blockWidth  = (40 > lineWidth - blockIndent) ? 40 : lineWidth - blockIndent;

   std::string line(blockWidth, '=');

   output << line.c_str() << "\n";
   
   return output;

   EXCEPTION_RETHROW;
}

}
