//*****************************************************************************
// FILE:        TestCase.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <sstream>
#include <iomanip>

#include "damkjerConfig.h"
#include "TestCase.h"

#include "Util/Streams/BlockIndent.h"

namespace Damkjer
{

//*****************************************************************************
// TestCase::TestCase
//*****************************************************************************
Util_API TestCase::TestCase(int argc, char** argv,
                   const std::string& caseName,
                   const std::string& caseID,
                   const std::string& caseDescription,
                   std::ostream& reportStream)
   : theArgc(argc)
   , theArgv(argv)
   , theCaseName(caseName)
   , theCaseID(caseID)
   , theCaseDescription(caseDescription)
   , theReportStream(reportStream)
   , theDetailsIndent(0)
   , theStartTime()
   , theStepCount(0)
   , theStreamFormatter(reportStream)
{
}
   
//*****************************************************************************
// TestCase::argv
//*****************************************************************************
Util_API std::string TestCase::argv(int i) const
{
   EXCEPTION_TRY("Damkjer::TestCase::argv(int)");

   if (i >= 0 && i < theArgc)
   {
      return theArgv[i];
   }

   //***
   // In released code, there's not a whole lot that the user can do about
   // this problem. This is a logic error and should be prevented through
   // appropriate bounds checking on the input prior to calling the method.
   //***
   std::ostringstream msg;

   msg << "Error encountered while retrieving an argument vector (argv) "
       << "element.\n\n"
       << "Unable to access the requested element (" << i << ") because it is "
       << "outside the valid bounds of [0, " << theArgc << ").";

   throw OutOfRange(msg.str(),  MODULE, __FILE__, __LINE__);

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::header
//*****************************************************************************
std::ostream& TestCase::header()
{
#if ( _WIN32 || _WIN64 )
   QueryPerformanceCounter(&theStartTime);
#elif ( __linux || __unix || __posix )
   clock_gettime(CLOCK_MONOTONIC, &theStartTime);
#else
   // How else can we capture high-resolution timing information?
#endif

   std::time_t now = std::time(NULL);

   theReportStream
      << dline
      << " TEST CASE   :  " << theCaseID << "\n"
      << " NAME        :  " << theCaseName << "\n"
      << "\n"
      << " DESCRIPTION :\n";

   // Indent this block by 4.
   theReportStream << blockIndent(4);
   theReportStream << theCaseDescription << "\n";
   theReportStream << endBlock;

   theReportStream
      << dline
      << sline
      << " START TIME  :  " << std::asctime(std::localtime(&now))
      << sline;

   return theReportStream;
}
   
//*****************************************************************************
// TestCase::footer
//*****************************************************************************
std::ostream& TestCase::footer()
{
   double elapsed = 0;

#if ( _WIN32 || _WIN64 )
   TimeT stopTime;
   QueryPerformanceCounter(&stopTime);

   TimeT frequency;
   QueryPerformanceFrequency(&frequency);

   elapsed = 1.0 * (stopTime.QuadPart - theStartTime.QuadPart) /
             frequency.QuadPart;
#elif ( __linux || __unix || __posix )
   TimeT stopTime;
   clock_gettime(CLOCK_MONOTONIC, &stopTime);

   elapsed = (stopTime.tv_sec - theStartTime.tv_sec) +
             (stopTime.tv_nsec - theStartTime.tv_nsec) * 1e-9;
#endif

   unsigned int iElapsed = unsigned int(elapsed);

   std::stringstream msg;
   
   char cfill = msg.fill('0');

   msg << std::setw(2) << iElapsed / 3600 << ":"
       << std::setw(2) << ((iElapsed / 60) % 60) << ":"
       << std::setw(9) << std::fixed << std::setprecision(6)
       << (elapsed-iElapsed+(iElapsed % 60));
   
   msg.fill(cfill);

   std::time_t now = std::time(NULL);

   theReportStream
//      << std::string(79,'-') << "\n"
      << sline
      << " STOP TIME   :  " << std::asctime(localtime(&now))
      << sline
      << dline
      << " ELAPSED     :  " << msg.str() << "\n"
      << dline;

   return theReportStream;
}

//*****************************************************************************
// TestCase::steps
//*****************************************************************************
int TestCase::steps()
{
   EXCEPTION_TRY("Damkjer::TestCase::steps()");

   bool testPassed = false;
   int passedTests = 0;

   for (int i = 0; i < theSteps.size(); ++i)
   {
      testing(theSteps[i]->description());
      theReportStream << blockIndent(theDetailsIndent);
      testPassed = (*(theSteps[i]))(*this);
      theReportStream << "\n" << ((testPassed) ? "[PASSED]" : "[FAILED]")
                      << "\n" << endBlock;

      if (testPassed) ++passedTests;

      theReportStream << sline;
   }
   
   theReportStream << "Summary: " << passedTests << "/" << theSteps.size()
                   << " tests passed.\n";

   return (passedTests == theSteps.size()) ? EXIT_SUCCESS : EXIT_FAILURE;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::execute
//*****************************************************************************
Util_API int TestCase::execute()
{
   EXCEPTION_TRY("Damkjer::TestCase::execute()");

   header();

   theReportStream << blockIndent(4);
   // Indent the body of the test report by 4.
   int results = steps();
   theReportStream << endBlock;

   footer();

   return results;

   EXCEPTION_RETHROW;
}

//*****************************************************************************
// TestCase::testing
//*****************************************************************************
void TestCase::testing(const std::string& description)
{
   std::ostringstream oss;
   oss << ++theStepCount << ". ";
   
   theDetailsIndent = static_cast<long>(oss.str().length());

   theReportStream << oss.str() << "Testing " << description << "...\n";

   return;
}


//*****************************************************************************
// TestCase::report
//*****************************************************************************
Util_API std::ostream& TestCase::report()
{
   return theReportStream;
}

//*****************************************************************************
// TestCase::registerStep
//*****************************************************************************
Util_API std::size_t TestCase::registerStep(TestStep* step)
{
   theSteps.push_back(step);
   return theSteps.size();
}

}