//*****************************************************************************
// FILE:        Exception.cpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-27  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#include <iostream>
#include <sstream>

#include "Exception.h"

namespace Damkjer
{

//*****************************************************************************
// Exception::~Exception
//*****************************************************************************
Util_API Exception::~Exception() throw()
{
}

//*****************************************************************************
// Exception::what
//*****************************************************************************
Util_API const char* Exception::what() const throw()
{
   if (theMsg.empty())
   {
      std::ostringstream oss;
      message(oss);
      theMsg = oss.str();
   }

   return theMsg.c_str();
}

//*****************************************************************************
// Exception::message
//*****************************************************************************
Util_API std::ostream& Exception::message(std::ostream& os) const
{
   os << "-- " << type() << " --\n";

   insertReason(os);

   os << "\n(" << who()
      << ", \"" << file() << "\""
      << ", line " << line() << ")\n";

   return os;
}

//*****************************************************************************
// Exception::push
//*****************************************************************************
Util_API void Exception::push(const std::string& who,
                     const std::string& file,
                     const int          line) const
{
   std::ostringstream os;
   os << who << " (\"" << file << "\"" << ", line " << line << ")";
   theStack.push_back(os.str());
}
 
//*****************************************************************************
// Exception::stackTrace
//*****************************************************************************
Util_API std::ostream& Exception::stackTrace(std::ostream& os) const
{
   if (theStack.empty()) return os;

   os << "\nStack Trace:";
//   os << "\nStack Trace:";

//   BlockIndent block(79, os);
//   block.indentBy(3);

   for (std::vector<std::string>::const_iterator vsi = theStack.begin();
        vsi != theStack.end();
        ++vsi)
   {
//      block << "\n" << *vsi;
      os << "\n" << *vsi;
   }

//   block << "\n";
   os << "\n";

   return os;
}

//*****************************************************************************
// Exception::Exception
//*****************************************************************************
Util_API Exception::Exception(const std::string& why,
                     const std::string& who,
                     const std::string& file,
                     int                line)
   : theWhy(why)
   , theWho(who)
   , theFile(file)
   , theLine(line)
{
}

//*****************************************************************************
// Exception::insertReason
//*****************************************************************************
Util_API std::ostream& Exception::insertReason(std::ostream& os) const
{
   return os << "\n" << theWhy << "\n";
}

//*****************************************************************************
// LogicError::LogicError
//*****************************************************************************
LogicError::LogicError(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : Exception(why, who, file, line)
   , std::logic_error("")
{
}

//*****************************************************************************
// LogicError::what
//*****************************************************************************
const char* LogicError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// DomainError::DomainError
//*****************************************************************************
DomainError::DomainError(const std::string& why,
                         const std::string& who,
                         const std::string& file,
                         int                line)
   : LogicError(why, who, file, line)
   , std::domain_error("")
{
}

//*****************************************************************************
// DomainError::what
//*****************************************************************************
const char* DomainError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// InvalidArgument::InvalidArgument
//*****************************************************************************
InvalidArgument::InvalidArgument(const std::string& why,
                                 const std::string& who,
                                 const std::string& file,
                                 int                line)
   : LogicError(why, who, file, line)
   , std::invalid_argument("")
{
}

//*****************************************************************************
// InvalidArgument::what
//*****************************************************************************
const char* InvalidArgument::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// LengthError::LengthError
//*****************************************************************************
LengthError::LengthError(const std::string& why,
                         const std::string& who,
                         const std::string& file,
                         int                line)
   : LogicError(why, who, file, line)
   , std::length_error("")
{
}

//*****************************************************************************
// LengthError::what
//*****************************************************************************
const char* LengthError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// OutOfRange::OutOfRange
//*****************************************************************************
OutOfRange::OutOfRange(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : LogicError(why, who, file, line)
   , std::out_of_range("")
{
}

//*****************************************************************************
// OutOfRange::what
//*****************************************************************************
const char* OutOfRange::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// RethrowError::RethrowError
//*****************************************************************************
RethrowError::RethrowError(const std::exception& thrown,
                                    const std::string& who,
                                    const std::string& file,
                                    int line)
   : DomainError((*thrown.what() != '\0' ? thrown.what()
                                         : typeid(thrown).name()),
                 who, file, line)
{
}

//*****************************************************************************
// RuntimeError::RuntimeError
//*****************************************************************************
RuntimeError::RuntimeError(const std::string& why,
                           const std::string& who,
                           const std::string& file,
                           int                line)
   : Exception(why, who, file, line)
   , std::runtime_error("")
{
}

//*****************************************************************************
// RuntimeError::what
//*****************************************************************************
const char* RuntimeError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// RangeError::RangeError
//*****************************************************************************
RangeError::RangeError(const std::string& why,
                       const std::string& who,
                       const std::string& file,
                       int                line)
   : RuntimeError(why, who, file, line)
   , std::range_error("")
{
}

//*****************************************************************************
// RangeError::what
//*****************************************************************************
const char* RangeError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// OverflowError::OverflowError
//*****************************************************************************
OverflowError::OverflowError(const std::string& why,
                             const std::string& who,
                             const std::string& file,
                             int                line)
   : RuntimeError(why, who, file, line)
   , std::overflow_error("")
{
}

//*****************************************************************************
// OverflowError::what
//*****************************************************************************
const char* OverflowError::what() const throw()
{
   return Exception::what();
}

//*****************************************************************************
// UnderflowError::UnderflowError
//*****************************************************************************
UnderflowError::UnderflowError(const std::string& why,
                               const std::string& who,
                               const std::string& file,
                               int                line)
   : RuntimeError(why, who, file, line)
   , std::underflow_error("")
{
}

//*****************************************************************************
// UnderflowError::what
//*****************************************************************************
const char* UnderflowError::what() const throw()
{
   return Exception::what();
}

}

//*****************************************************************************
// ::operator<<
//*****************************************************************************
//Util_API std::ostream& operator<<(std::ostream& os,
//                                  const Damkjer::Exception& e)
//{
//   return os << e.what();
//}
