//*****************************************************************************
// FILE:        Exception.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-27  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef Exception_HEADER
#define Exception_HEADER

#include <typeinfo>
#include <exception>
#include <stdexcept>
#include <vector>
#include <string>

#include "Util/UtilAPI.h"

namespace Damkjer
{

//*****************************************************************************
// CLASS: Exception
//> Exception objects allow for detailed, robust handling of errors and
//  exceptional circumstances encountered in programs. This class, and its
//  derivatives, intnentionally mirror the C++ Standard Library exception
//  classes. They also provide for standardized message formatting and stack
//  trace aggregation through convenience macros.
//
//  The implemented exceptions that parallel the C++ Standard Library are:
//
//- Exception                ->     std::exception
//-    LogicError            ->        std::logic_error
//-       DomainError        ->           std::domain_error
//-       InvalidArgument    ->           std::invalid_argument
//-       LengthError        ->           std::length_error
//-       OutOfRange         ->           std::out_of_range
//-    RuntimeError          ->        std::runtime_error
//-       RangeError         ->           std::range_error
//-       OverflowError      ->           std::overflow_error
//-       UnderflowError     ->           std::underflow_error
//<
//*****************************************************************************
class Exception
{
public:
   Util_API virtual ~Exception() throw();
      //>
      //<

   Util_API virtual const char* what() const throw() = 0;
      //>
      //<

   Util_API virtual std::ostream& message(std::ostream&) const;
      //>
      //<

   Util_API void push(const std::string&,
             const std::string&,
             const int) const;
      //>
      //<

   Util_API std::ostream& stackTrace(std::ostream&) const;
      //>
      //<

   Util_API const std::string& type() const {
      if (theTypeID.empty())
      {
         theTypeID = typeid(*this).name();
      }
      return theTypeID;
   }
   Util_API const std::string& why()  const { return theWhy; }
   Util_API const std::string& who()  const { return theWho; }
   Util_API const std::string& file() const { return theFile; }
   Util_API const int          line() const { return theLine; }

protected:
   Util_API Exception(const std::string&,
             const std::string&,
             const std::string&,
             int);
      //>
      //<

   Util_API virtual std::ostream& insertReason(std::ostream&) const;
      //>
      //<

private:
   mutable std::string theTypeID;
   std::string theWhy;
   std::string theWho;
   std::string theFile;
   int         theLine;

   mutable std::string theMsg;

   mutable std::vector<std::string> theStack;
};

//*****************************************************************************
// CLASS: LogicError
//> 
//<
//*****************************************************************************
class LogicError : public Exception,
                   public std::logic_error
{
public:
   Util_API LogicError(const std::string&,
              const std::string&,
              const std::string&,
              int);
      //>
      //<

   Util_API virtual ~LogicError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: DomainError
//>
//<
//*****************************************************************************
class DomainError : public LogicError,
                    public std::domain_error
{
public:
   Util_API DomainError(const std::string&,
               const std::string&,
               const std::string&,
               int);
      //>
      //<

   Util_API virtual ~DomainError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: InvalidArgument
//>
//<
//*****************************************************************************
class InvalidArgument : public LogicError,
                                 public std::invalid_argument
{
public:
   Util_API InvalidArgument(const std::string&,
                   const std::string&,
                   const std::string&,
                   int);
      //>
      //<

   Util_API virtual ~InvalidArgument() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: LengthError
//>
//<
//*****************************************************************************
class LengthError : public LogicError,
                             public std::length_error
{
public:
   Util_API LengthError(const std::string&,
               const std::string&,
               const std::string&,
               int);
      //>
      //<

   Util_API virtual ~LengthError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: OutOfRange
//>
//<
//*****************************************************************************
class OutOfRange : public LogicError,
                            public std::out_of_range
{
public:
   Util_API OutOfRange(const std::string&,
              const std::string&,
              const std::string&,
              int);
      //>
      //<

   Util_API virtual ~OutOfRange() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: RethrowError
//>
//<
//*****************************************************************************
class RethrowError : public DomainError
{
public:
   Util_API RethrowError(const std::exception&,
                const std::string&,
                const std::string&,
                int);
      //> This constructor is used to wrap a std::exception as a
      //  Damkjer::Exception for the purpose of rethrowing. The what()
      //  portion of the std::exception is reported in the what()
      //  of the RethrowError.
      //<

   Util_API RethrowError(const Damkjer::Exception& thrown,
                const std::string&        extraReason,
                const std::string&        who,
                const std::string&        file,
                int                       line)
      : DomainError(thrown.why() + " " + extraReason, who, file, line)
   {
   }
      //> This constructor is used to add an extra reason to a
      //  Damkjer::Exception for the purpose of rethrowing.
      //<

   Util_API RethrowError(const std::string& why,
                const std::string& who,
                const std::string& file,
                int                line)
      : DomainError(why, who, file, line)
   {
   }
      //> This constructor is used to wrap a string as a
      //  Damkjer::Exception for the purpose of rethrowing. The what()
      //  portion of the std::exception is reported in the what()
      //  of the RethrowError.
      //<

   Util_API RethrowError(const std::string& who,
                const std::string& file,
                int                line)
      : DomainError("Unknown error", who, file, line)
   {
   }
      //> This constructor is used to wrap an unknown exception as a
      //  Damkjer::Exception for the purpose of rethrowing. The message
      //  "Unknown error" is reported in the what() of the RethrowError.
      //<
};

//*****************************************************************************
// CLASS: RuntimeError
//>
//<
//*****************************************************************************
class RuntimeError : public Exception,
                              public std::runtime_error
{
public:
   Util_API RuntimeError(const std::string&,
                const std::string&,
                const std::string&,
                int);
      //>
      //<

   Util_API virtual ~RuntimeError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: RangeError
//>
//<
//*****************************************************************************
class RangeError : public RuntimeError,
                            public std::range_error
{
public:
   Util_API RangeError(const std::string&,
              const std::string&,
              const std::string&,
              int);
      //>
      //<

   Util_API virtual ~RangeError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: OverflowError
//>
//<
//*****************************************************************************
class OverflowError : public RuntimeError,
                               public std::overflow_error
{
public:
   Util_API OverflowError(const std::string&,
                 const std::string&,
                 const std::string&,
                 int);
      //>
      //<

   Util_API virtual ~OverflowError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

//*****************************************************************************
// CLASS: UnderflowError
//>
//<
//*****************************************************************************
class UnderflowError : public RuntimeError,
                                public std::underflow_error
{
public:
   Util_API UnderflowError(const std::string&,
                  const std::string&,
                  const std::string&,
                  int);
      //>
      //<

   Util_API virtual ~UnderflowError() throw() {}
      //>
      //<

   Util_API virtual const char* what() const throw();
      //>
      //<
};

}

Util_API inline std::ostream& operator<<(std::ostream& os, const Damkjer::Exception& e)
{
   return os << e.what();
}

//*****************************************************************************
// MACRO: EXCEPTION_TRY
//>
//<
//*****************************************************************************
#define EXCEPTION_TRY(MODULE_NAME) \
static const char* const MODULE = MODULE_NAME; \
try \
{

//*****************************************************************************
// MACRO: EXCEPTION_RETHROW
//>
//<
//*****************************************************************************
#define EXCEPTION_RETHROW \
} \
catch (const Damkjer::Exception& e) \
{ \
   e.push(MODULE, __FILE__, __LINE__); \
   throw; \
} \
catch (const std::exception& e) \
{ \
   throw RethrowError(e, MODULE, __FILE__, __LINE__); \
} \
catch (...) \
{ \
   throw RethrowError(MODULE, __FILE__, __LINE__); \
}

//*****************************************************************************
// MACRO: EXCEPTION_CATCHALL
//>
//<
//*****************************************************************************
#define EXCEPTION_CATCHALL \
} \
catch (const Damkjer::Exception& e) \
{ \
   e.push(MODULE, __FILE__, __LINE__); \
   std::cerr << e; \
   e.stackTrace(std::cerr); \
   return EXIT_FAILURE; \
} \
catch (const std::exception& e) \
{ \
   std::cerr << e.what() << std::endl; \
   return EXIT_FAILURE; \
} \
catch (...) \
{ \
   std::cerr << "Unknown exception" << std::endl; \
   return EXIT_FAILURE; \
}

#endif