//*****************************************************************************
// FILE:        Metrics.h
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS:
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Modified free functions to functors to allow for "functions
//               with state" like p-norm.
//<
//*****************************************************************************

#ifndef Metrics_HEADER
#define Metrics_HEADER

#include <cmath>
#include <iterator>
#include <sstream>

#include "damkjerConfig.h"

namespace Damkjer
{
//*****************************************************************************
// FUNCTOR: EuclideanDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = double>
class EuclideanDistance
{
public:
   typedef ReturnT value_type;
   
   value_type operator()(const PointT& a, const PointT& b) const
   {
      value_type sum = 0;

      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         sum += (*ai - *bi) * (*ai - *bi);
      }

      return std::sqrt(sum);
   }
};

//*****************************************************************************
// FUNCTOR: ManhattanDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = double>
class ManhattanDistance
{
public:
   typedef ReturnT value_type;

   value_type operator()(const PointT& a, const PointT& b) const
   {
      value_type sum = 0;
      
      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         sum += std::fabs(*ai - *bi);
      }
      
      return sum;
   }
};

//*****************************************************************************
// FUNCTOR: ChebyshevDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = double>
class ChebyshevDistance
{
public:
   typedef ReturnT value_type;

   value_type operator()(const PointT& a, const PointT& b) const
   {
      value_type my_max = 0;
      
      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         value_type temp = std::fabs(*ai - *bi);
         my_max = (temp > my_max) ? temp : my_max;
      }
      
      return my_max;
   }
};

//*****************************************************************************
// FUNCTOR: MinkowskiDistance
//*****************************************************************************
template<typename PointT, typename ReturnT = double>
class MinkowskiDistance
{
public:
   typedef ReturnT value_type;

   MinkowskiDistance() : theP(2){}

   explicit MinkowskiDistance(value_type p) : theP(p)
   {
      EXCEPTION_TRY("Damkjer::MinkowskiDistance<PointT, ReturnT>::MinkowskiDistance");

      if (theP < 1)
      {
         //***
         // In released code, there's not a whole lot that the user can do
         // about this problem. This is a logic error and should be prevented
         // through appropriate bounds checking on the input prior to
         // calling the constructor.
         //***
         std::ostringstream msg;

         msg << "Context : Defining a Minkowski distance metric.\n"
             << "Problem : Unable to define a Minkowski distance metric with "
             << "the requested p-value.\n"
             << "Cause   : Minkowski distance must be defined with p >= 1 to "
             << "describe a metric space. p-value requested: " << p << "\n"
             << "Solution: Restrict p-values to p >= 1 if possible. "
             << "Otherwise, contact support for assitance.";

         throw DomainError(msg.str(),  MODULE, __FILE__, __LINE__);
      }

      EXCEPTION_RETHROW;
   }

   value_type operator()(const PointT& a, const PointT& b) const
   {
      value_type sum = 0;
   
      for (typename PointT::const_iterator ai = a.begin(), bi = b.begin();
           ai != a.end() && bi != b.end();
           ++ai, ++bi)
      {
         sum += std::pow(static_cast<value_type>(std::fabs(*ai - *bi)), theP);
      }
   
      return std::pow(sum, static_cast<value_type>(1./theP));
   }

protected:
   value_type theP;
};

}

#endif
