//*****************************************************************************
// FILE:        VpTree.cpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION: This class is an implementation of the vantage point tree
//              data structure described by Peter Yianilos in "Data
//              Structures and Algorithms for Nearest Neighbor Search in
//              General Metric Spaces".
//
// LIMITATIONS: See VpTree.h for full list of limitations.
//
// SOFTWARE HISTORY:
//> 2012-SEP-11  K. Damkjer
//               Initial Coding.
//  2013-JUL-23  K. Damkjer
//               Set typedefs to make code more readable and to allow for
//               varying container and metric types. This is useful when the
//               default double-precision is overkill. It is now easy to set
//               types to float.
//<
//*****************************************************************************

#if _OPENMP
#include <omp.h>
#endif

#include "VpTree.h"

#include <algorithm>
#include <iostream>

namespace Damkjer
{

//*****************************************************************************
// CLASS: VpTree::Node
//*****************************************************************************
template<typename PointT, typename MetricT>
class VpTree<PointT, MetricT>::Node
{
public:
   Node();
      //> The default constructor creates a null node.
      //<
    
   Node(VpTree*);
      //> Construct with a database reference.
      //<
    
   virtual ~Node();
      //> The default destructor. Virtual to ensure proper Node
      //  destruction.
      //<
    
   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const = 0;
      //> The visitor that accumulates k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const = 0;
      //> The visitor that accumulates fixed-radius nearest neighbor
      //  results.
      //<

protected:
   VpTree<PointT, MetricT>* theTree;
      //> The tree that this node belongs to.  Included to avoid reallocating
      //  contained element data for each node. As a side effect, nodes
      //  organize the tree's data contents through the construction process.
      //
      //  I personally debated including this pointer for quite a while since
      //  it seemed like a bad design to have such a cyclic reference in a tree
      //  structure, but it appears to be quite beneficial from a performance
      //  perspective.
      //<
};

//*****************************************************************************
// CLASS: VpTree::Internal
//*****************************************************************************
template<typename PointT, typename MetricT>
class VpTree<PointT, MetricT>::Internal : public VpTree<PointT, MetricT>::Node
{
public:
   Internal();
      //> The default constructor creates a null internal node.
      //<
    
   Internal(VpTree*);
      //> Construct with a database reference.
      //<
    
   virtual ~Internal();
      //> The default destructor. Virtual to ensure proper Node
      //  destruction.
      //<
    
   IndexT theIndex;
      //> The index of the item in the internal database.
      //<

   DistT theInnerLowerBound;
      //> The lower bound distance to elements on inner branch.
      //<

   DistT theInnerUpperBound;
      //> The uuper bound distance to elements on inner branch.
      //<

   DistT theOuterLowerBound;
      //> The lower bound distance to elements on outer branch.
      //<

   DistT theOuterUpperBound;
      //> The upper bound distance to elements on outer branch.
      //<

   Node* theInnerBranch;
      //> The inner branch partition containing elements closer than
      //  theMedianPartition to this element.
      //<

   Node* theOuterBranch;
      //> The outer branch partition containing elements at or beyond
      //  theMedianPartition from this element.
      //<

   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const;
      //> The visitor that accumulates k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const;
      //> The visitor that accumulates fixed radius nearest neighbor
      //  results.
      //<
};

//*****************************************************************************
// CLASS: VpTree::Leaf
//*****************************************************************************
template<typename PointT,
         typename MetricT>
class VpTree<PointT, MetricT>::Leaf : public VpTree<PointT, MetricT>::Node
{
public:
   Leaf();
      //> The default constructor creates a null leaf.
      //<
    
   Leaf(VpTree*, const IndexT&, const IndexT&);
      //> Construct a leaf representing the items in the provided index
      //  range.
      //<
    
   virtual ~Leaf(){}
      //> The default destructor. Virtual to ensure proper Node
      //  destruction.
      //<
    
   virtual void knn(const PointT&,
                    const IndexT&,
                    ResultsSetT&,
                    DistT&) const;
      //> The visitor that accumulates k nearest neighbor results.
      //<

   virtual void rnn(const PointT&,
                    const DistT&,
                    ResultsSetT&) const;
      //> The visitor that accumulates fixed radius nearest neighbor
      //  results.
      //<

private:
   IndexT theHead;
      //> The start of the index range prepresented by this Leaf.
      //<

   IndexT theTail;
      //> The end of the index range prepresented by this Leaf.
      //<
};

//*****************************************************************************
// CLASS: VpTree::Item
//*****************************************************************************
template<typename PointT, typename MetricT>
class VpTree<PointT, MetricT>::Item
{
public:
   Item();
      //> The default constructor creates a null node.
      //<
    
   ~Item(){}
      //> The default destructor. Intentionally non-virtual since Node is
      //  a private inner class on VpTree.
      //<
    
   bool operator<(const Item&) const;
      //>
      //<

   IndexT theIndex;
      //> The index of the item in the input data set.
      //<

   PointT theElement;
      //> The database object
      //<
   
   DistT theDistance;
      //> The most recent ancestral pivot history distance for this item.
      //<
};

//*****************************************************************************
// VpTree::ResultsCandidate
//*****************************************************************************
template<typename PointT, typename MetricT>
class VpTree<PointT, MetricT>::ResultsCandidate
{
public:
   ResultsCandidate(const IndexT&, const DistT&);
      //> Create a metric comparitor.
      //<
    
   ~ResultsCandidate(){}
      //> The default destructor. Intentionally non-virtual since
      //  ResultsCandidate is a private inner class on VpTree.
      //<

   bool operator<(const ResultsCandidate&) const;
      //> Compare result candidate distances to determine which is closer
      //  to the query.
      //<

   IndexT theIndex;
      //> The index of the candidate in the internal database.
      //<
    
   DistT theDistance;
      //> The candidate's distance to the query point.
      //<
};

//*****************************************************************************
// VpTree::VpTree
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::VpTree(const MetricT& metric,
                                       const IndexT& leafCapacity)
   : theRoot(0)
   , theItems()
   , theLeafCapacity((leafCapacity<1)?1:leafCapacity)
   , theMetric(metric)
{
}

//*****************************************************************************
// VpTree::VpTree
//*****************************************************************************
template<typename PointT, typename MetricT>
template<typename DatabaseT>
inline VpTree<PointT, MetricT>::VpTree(const DatabaseT& elems,
                                       const MetricT& metric,
                                       const IndexT& leafCapacity)
   : theRoot(0)
   , theItems(elems.size())
   , theLeafCapacity((leafCapacity<1)?1:leafCapacity)
   , theMetric(metric)
{
   #if _OPENMP
   omp_set_dynamic(1);
   omp_set_num_threads(omp_get_num_procs());
   #endif

   #pragma omp parallel for
   for (int i = 0; i < theItems.size(); ++i)
   {
       theItems[i].theIndex=i;
       theItems[i].theElement=elems[i];
   }

   theRoot = makeTree(0, theItems.size());
}
                
//*****************************************************************************
// VpTree::~VpTree
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::~VpTree()
{
   delete theRoot;
   theRoot = 0;
}

//*****************************************************************************
// VpTree::Node::Node
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Node::Node()
   : theTree(0)
{
}

//*****************************************************************************
// VpTree::Node::Node
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Node::Node(VpTree* tree)
   : theTree(tree)
{
}

//*****************************************************************************
// VpTree::Node::~Node
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Node::~Node()
{
   theTree = 0;
}

//*****************************************************************************
// VpTree::Internal::Internal
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Internal::Internal()
   : Node()
   , theIndex(0)
   , theInnerLowerBound(0)
   , theInnerUpperBound(0)
   , theOuterLowerBound(0)
   , theOuterUpperBound(0)
   , theInnerBranch(0)
   , theOuterBranch(0)
{
}

//*****************************************************************************
// VpTree::Internal::Internal
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Internal::Internal(VpTree<PointT, MetricT>* tree)
   : Node(tree)
   , theIndex(0)
   , theInnerLowerBound(0)
   , theInnerUpperBound(0)
   , theOuterLowerBound(0)
   , theOuterUpperBound(0)
   , theInnerBranch(0)
   , theOuterBranch(0)
{
}

//*****************************************************************************
// VpTree::Internal::~Internal
//*****************************************************************************
template<typename PointT,
         typename MetricT>
inline VpTree<PointT, MetricT>::Internal::~Internal()
{
   delete theInnerBranch;
   delete theOuterBranch;
   theInnerBranch = 0;
   theOuterBranch = 0;
}

//*****************************************************************************
// VpTree::Leaf::Leaf
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Leaf::Leaf()
   : Node()
   , theHead(0)
   , theTail(0)
{
}

//*****************************************************************************
// VpTree::Leaf::Leaf
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Leaf::Leaf(VpTree* tree,
                                           const IndexT& head,
                                           const IndexT& tail)
   : Node(tree)
   , theHead(head)
   , theTail(tail)
{
}

//*****************************************************************************
// VpTree::Item::Item
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::Item::Item()
   : theIndex(0)
   , theElement(0)
   , theDistance(0)
{
}

//*****************************************************************************
// VpTree::Item::operator<
//*****************************************************************************
template<typename PointT, typename MetricT>
inline bool VpTree<PointT, MetricT>::Item::operator<(const Item& other) const
{
   return theDistance < other.theDistance;
}

//*****************************************************************************
// VpTree::ResultsCandidate::ResultsCandidate
//*****************************************************************************
template<typename PointT, typename MetricT>
inline VpTree<PointT, MetricT>::ResultsCandidate::ResultsCandidate(
                                                      const IndexT& index,
                                                      const DistT& distance)
   : theIndex(index)
   , theDistance(distance)
{
}

//*****************************************************************************
// VpTree::ResultsCandidate::operator<
//*****************************************************************************
template<typename PointT, typename MetricT>
inline bool VpTree<PointT, MetricT>::ResultsCandidate::operator<(
                                           const ResultsCandidate& other) const
{
   return theDistance < other.theDistance;
}

//*****************************************************************************
// VpTree::randomSample
//*****************************************************************************
template<typename PointT, typename MetricT>
std::set<typename VpTree<PointT, MetricT>::IndexT>
VpTree<PointT, MetricT>::randomSample(const IndexT& start,
                                      const IndexT& stop) const
{
   //***
   // Sampling the sqrt of inputs, while thorough, is completely unnecessary.
   // Leaving the note here for future reference.
   //***
   // IndexT numSamps=(IndexT)(ceil(sqrt((double)(stop - start))));

   // A very small sample set of the population is sufficient
   IndexT numSamps=(stop-start > 5) ? 5 : (stop - start);

   //***
   // If the range is smaller than the number of samples, just return the
   // elements in the range.
   //***
   if ((stop - start) <= numSamps)
   {
      std::vector<IndexT> indices(stop-start, 0);

      for (std::size_t i=start; i < stop; ++i)
      {
         indices[i-start]=i;
      }
        
      return std::set<IndexT>(indices.begin(), indices.end());
   }
    
   std::set<IndexT> samples;
    
   //***
   // If the range is close to the number of samples, select with better
   // worst-case behvior
   //***
   if ((stop - start) < numSamps*2)
   {
      IndexT itemsNeeded=numSamps;
        
      for (IndexT i = start; samples.size() < numSamps && i < stop; ++i)
      {
         if ((rand()/(RAND_MAX + 1.0)) < itemsNeeded/(stop-i))
         {
            samples.insert(i);
            --itemsNeeded;
         }
      }
   }
   else
   {
      //***
      // Otherwise, if range dominates samples, select expecting to find
      // unique samples
      //***
      while (samples.size() < numSamps)
      {
         // Choose an arbitrary point
         IndexT val=static_cast<IndexT>(rand() / (RAND_MAX + 1.0) *
                                        (stop - start) + start);

         samples.insert(val);
      }
   }
   
   return samples;
}
                                
//*****************************************************************************
// VpTree::selectVp
//*****************************************************************************
template<typename PointT, typename MetricT>
typename VpTree<PointT, MetricT>::IndexT
VpTree<PointT, MetricT>::selectVp(const IndexT& start,
                                  const IndexT& stop) const
{
   //***
   // Choosing a vantage point that maximizes the balance of the sub-trees is
   // theoretically advantageous. This involves selecting a vantage point in a 
   // "corner" of the population. However, in practice, the trade for increased
   // selection time dwarfs the benefit realized through search times. The
   // incredibly simple approach of selecting a random member of the population
   // is much simpler and yields almost identical search times in the data sets
   // tested.
   //
   // The "intellegent" selection mode is included in the source code for
   // reference, but dropped by the pre-processor.
   //***

#if 0
   // Choose a point from a small sample set that maximizes spread 
   std::set<IndexT> p=randomSample(start, stop);
    
   IndexT bestP=*(p.begin());
   DistT bestSpread=0;
    
   for (std::set<IndexT>::const_iterator pItr=p.begin();
        pItr != p.end();
        ++pItr)
   {
      const VpTree<PointT, MetricT>::Item& pItem = theItems[*pItr];

      std::set<IndexT> d=randomSample(start, stop);
      
      std::vector<VpTree<PointT, MetricT>::Item> dItems(d.size());
      
      IndexT i = 0;
      
      for (std::set<IndexT>::const_iterator dItr=d.begin();
           dItr != d.end();
           ++dItr)
      {
          dItems[i]=theItems[*dItr];
          ++i;
      }

      std::nth_element(dItems.begin(),
                       dItems.begin() + dItems.size()/2,
                       dItems.end());
      
      DistT mu = theMetric(pItem.theElement,
                           dItems[dItems.size()/2].theElement);

      IndexT k=1;
      DistT x, oldM, newM;

      x=oldM=newM=theMetric(pItem.theElement,
                            dItems[0].theElement)-mu;
      
      DistT oldS, newS;
      oldS=newS=0;
      
      for (IndexT i = 1; i < dItems.size(); ++i)
      {
         x=theMetric(pItem.theElement,
                     dItems[i].theElement)-mu;

         ++k;
         newM=oldM+(x-oldM)/k;
         newS=oldS+(x-oldM)*(x-newM);
         oldM=newM;
         oldS=newS;
      }
      
      DistT spread=static_cast<DistT>((k>1)?newS/(k-1):0.0);
      
      if (spread > bestSpread)
      {
          bestSpread=spread;

          bestP=*pItr;
      }
   }
   
   return bestP;
#else
   // Simplest working case: just choose an arbitrary point
   return static_cast<IndexT>(rand()/(RAND_MAX + 1.0)*(stop-start)+start);
#endif
}

//*****************************************************************************
// VpTree::makeTree
//*****************************************************************************
template<typename PointT, typename MetricT>
typename VpTree<PointT, MetricT>::Node*
VpTree<PointT, MetricT>::makeTree(const IndexT& start, const IndexT& stop)
{
   if (stop <= start) return 0;

   IndexT setSize = stop - start;
   
   if (setSize <= theLeafCapacity)
   {
      return new Leaf(this, start, stop);
   }
 
   Internal* node=new Internal(this);
   node->theIndex=start;

   const IndexT vp = selectVp(start, stop);
   std::swap(theItems[start], theItems[vp]);
   
   // Identify bound elements
   IndexT outerLowerBound = (start + stop + 1)/2;
   IndexT innerLowerBound = start + 1;
   
   // Update histories
   DistT d_max=0;
   DistT d_min=std::numeric_limits<DistT>::max();
   for (IndexT elem = stop; elem --> innerLowerBound; )
   {
      theItems[elem].theDistance = theMetric(theItems[start].theElement,
                                             theItems[elem].theElement);
      d_max = (theItems[elem].theDistance > d_max)
              ? theItems[elem].theDistance : d_max;
      d_min = (theItems[elem].theDistance < d_min)
              ? theItems[elem].theDistance : d_min;
   }
   
   // Put the median element in place
   std::nth_element(
           theItems.begin() + innerLowerBound,
           theItems.begin() + outerLowerBound,
           theItems.begin() + stop);

   DistT d_mid=0;
   for (IndexT elem = outerLowerBound; elem --> innerLowerBound; )
   {
      d_mid = (theItems[elem].theDistance > d_mid)
              ? theItems[elem].theDistance : d_mid;
   }
   
   node->theInnerLowerBound=d_min;
   node->theInnerUpperBound=d_mid;
   node->theOuterLowerBound=theItems[outerLowerBound].theDistance;
   node->theOuterUpperBound=d_max;
   
   #if _OPENMP
   static int threads = 0;
   
   if (threads < (omp_get_num_procs()-1))
   {
      omp_set_dynamic(1);
      omp_set_num_threads(omp_get_num_procs());
      omp_set_nested(1);
      
      #pragma omp atomic
      ++threads;
      
      #pragma omp parallel
      {         
         #pragma omp sections nowait
         {
            node->theInnerBranch = makeTree(innerLowerBound,
                                            outerLowerBound);
              
            #pragma omp section
            node->theOuterBranch = makeTree(outerLowerBound, stop);
         }
      }
      
      #pragma omp atomic
      --threads;
   }
   else
   #endif
   {
      node->theInnerBranch = makeTree(innerLowerBound, outerLowerBound);
      node->theOuterBranch = makeTree(outerLowerBound, stop);
   }
   
   return node;
}

//*****************************************************************************
// VpTree::knn
//*****************************************************************************
template<typename PointT, typename MetricT>
typename VpTree<PointT, MetricT>::SearchResultsT
VpTree<PointT, MetricT>::knn(const PointT& query,
                             const IndexT& k,
                             const DistT& limit) const
{
   ResultsSetT candidates;
   DistT tau = limit;

   theRoot->knn(query, k, candidates, tau);

   std::deque<IndexT> indices;
   std::deque<DistT> distances;


   while(!candidates.empty())
   {
      indices.push_back(theItems[candidates.top().theIndex].theIndex);
      distances.push_back(candidates.top().theDistance);
      candidates.pop();
   }

   std::reverse(indices.begin(), indices.end());
   std::reverse(distances.begin(), distances.end());

   return std::make_pair(indices, distances);
}

//*****************************************************************************
// VpTree::Internal::knn
//*****************************************************************************
template<typename PointT, typename MetricT>
void
VpTree<PointT, MetricT>::Internal::knn(
        const PointT& query,
        const IndexT& k,
        ResultsSetT& candidates,
        DistT& kth_closest) const
{
   if (!(theTree)) return;

   DistT distance = theTree->theMetric(theTree->theItems[theIndex].theElement,
                                       query);

   if (distance < kth_closest)
   {
      if (candidates.size() == k)
      {
          candidates.pop();
      }
      
      candidates.push(ResultsCandidate(theIndex, distance));
      
      if (candidates.size() == k)
      {
          kth_closest = candidates.top().theDistance;
      }
   }

   if (!(theInnerBranch || theOuterBranch))
   {
       return;
   }

   DistT middle = static_cast<DistT>
                  (0.5 * (theInnerUpperBound + theOuterLowerBound));

   if (distance < middle)
   {
      if (theInnerBranch &&
          (distance - theInnerUpperBound <= kth_closest) &&
          (theInnerLowerBound - distance <= kth_closest))


      if (theInnerBranch &&
          (distance <= theInnerUpperBound + kth_closest) &&
          (distance >= theInnerLowerBound - kth_closest))
      {
         theInnerBranch->knn(query, k, candidates, kth_closest);
      }
       
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - kth_closest) &&
          (distance <= theOuterUpperBound + kth_closest))
      {
         theOuterBranch->knn(query, k, candidates, kth_closest);
      }
   }
   else
   {
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - kth_closest) &&
          (distance <= theOuterUpperBound + kth_closest))
      {
         theOuterBranch->knn(query, k, candidates, kth_closest);
      }
       
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + kth_closest) &&
          (distance >= theInnerLowerBound - kth_closest))
      {
         theInnerBranch->knn(query, k, candidates, kth_closest);
      }
   }
}

//*****************************************************************************
// VpTree::Leaf::knn
//*****************************************************************************
template<typename PointT, typename MetricT>
void
VpTree<PointT, MetricT>::Leaf::knn(
        const PointT& query,
        const IndexT& k,
        ResultsSetT& candidates,
        DistT& kth_closest) const
{
   if (!(theTree)) return;
   
   // Scan the leaf
   for (IndexT item = theHead; item < theTail; ++item)
   {
      // This is the check state
      DistT distance = theTree->theMetric(theTree->theItems[item].theElement,
                                          query);
      
      if (distance < kth_closest)
      {
         if (candidates.size() == k)
         {
            candidates.pop();
         }
         
         candidates.push(ResultsCandidate(item, distance));
         
         if (candidates.size() == k)
         {
            kth_closest = candidates.top().theDistance;
         }
      }
   }
}

//*****************************************************************************
// VpTree::rnn
//*****************************************************************************
template<typename PointT, typename MetricT>
typename VpTree<PointT, MetricT>::SearchResultsT
VpTree<PointT, MetricT>::rnn(const PointT& query,
                             const DistT& range) const
{
   ResultsSetT candidates;

   theRoot->rnn(query, range, candidates);

   std::deque<IndexT> indices;
   std::deque<DistT> distances;

   while(!candidates.empty())
   {
      indices.push_back(theItems[candidates.top().theIndex].theIndex);
      distances.push_back(candidates.top().theDistance);
      candidates.pop();
   }

   std::reverse(indices.begin(), indices.end());
   std::reverse(distances.begin(), distances.end());

   return std::make_pair(indices, distances);
}

//*****************************************************************************
// VpTree::Internal::rnn
//*****************************************************************************
template<typename PointT, typename MetricT>
void VpTree<PointT, MetricT>::Internal::rnn(const PointT& query,
                                            const DistT& range,
                                            ResultsSetT& candidates) const
{
   if (!(theTree)) return;
   
   DistT distance = theTree->theMetric(theTree->theItems[theIndex].theElement,
                                       query);

   if (distance <= range)
   {
      candidates.push(ResultsCandidate(theIndex, distance));
   }

   if (!(theInnerBranch || theOuterBranch))
   {
       return;
   }

   DistT middle = static_cast<DistT>
                  (0.5 * (theInnerUpperBound + theOuterLowerBound));

   if (distance < middle)
   {
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + range) &&
          (distance >= theInnerLowerBound - range))
      {
         theInnerBranch->rnn(query, range, candidates);
      }
       
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - range) &&
          (distance <= theOuterUpperBound + range))
      {
         theOuterBranch->rnn(query, range, candidates);
      }
   }
   else
   {
      if (theOuterBranch &&
          (distance >= theOuterLowerBound - range) &&
          (distance <= theOuterUpperBound + range))
      {
         theOuterBranch->rnn(query, range, candidates);
      }
       
      if (theInnerBranch &&
          (distance <= theInnerUpperBound + range) &&
          (distance >= theInnerLowerBound - range))
      {
         theInnerBranch->rnn(query, range, candidates);
      }
   }
}

//*****************************************************************************
// Leaf::rnn
//*****************************************************************************
template<typename PointT, typename MetricT>
void VpTree<PointT, MetricT>::Leaf::rnn(
        const PointT& query,
        const DistT& range,
        ResultsSetT& candidates) const
{
   if (!(theTree)) return;

   // Scan the leaf
   for (IndexT item = theHead; item < theTail; ++item)
   {
      DistT distance = theTree->theMetric(theTree->theItems[item].theElement,
                                          query);
      
      if (distance <= range)
      {
         candidates.push(ResultsCandidate(item, distance));
      }
   }
}

}
