//*****************************************************************************
// FILE:        UnitTest.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION: 
//
// LIMITATIONS: 
//
// SOFTWARE HISTORY:
//> 2012-JUL-24  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#ifndef TestCase_HEADER
#define TestCase_HEADER

#if ( _WIN32 || _WIN64 )
#include <windows.h>
#elif ( __linux || __unix || __posix )
#include <time.h>
#endif

#include <cassert>
#include <iostream>
#include <string>
#include <ctime>
#include <vector>

#include "Util/UtilAPI.h"
#include "Util/Streams/FormatFilter.h"

namespace Damkjer
{

//*****************************************************************************
// CLASS: TestCase
//*****************************************************************************
class TestCase
{
public:
#if ( _WIN32 || _WIN64 )
   typedef LARGE_INTEGER TimeT;
#elif ( __linux || __unix || __posix )
   typedef timespec TimeT;
#else
   // How else can we capture high-resolution timing information?
#endif

   Util_API TestCase(int argc, char** argv,
                     const std::string& caseName,
                     const std::string& caseID,
                     const std::string& caseDescription,
                     std::ostream& reportStream=std::clog);
      //> This constructor sets the name, ID, and description for the unit test
      //  case. It also wraps an output stream with a report stream to easily
      //  redirect and format test logs.
      //<
   
   Util_API virtual ~TestCase(){}
      //> Free resources owned by this class.
      //<

   Util_API virtual int execute();
      //> Execute the test case. This method performs the simple sequence:
      //  1) Output Header to the report stream
      //  2) Run steps for the test
      //  3) Output Footer to the report stream
      //<

   Util_API virtual std::ostream& report();
      //>
      //<

   class TestStep
   {
   public:
      Util_API TestStep(const std::string& description)
         : theDescription(description)
      {
      }

      Util_API const std::string& description() const { return theDescription; }

      Util_API virtual bool operator()(TestCase&) const = 0;

   private:
      std::string theDescription;
   };

   Util_API std::size_t registerStep(TestStep* step);

   Util_API int stepNum() const { return theStepCount; }

   Util_API int argc() const { return theArgc; }
      //> The command line argument count.
      //<
   
   Util_API std::string argv(int i) const;
      //> The requested command line argument, converted to a string.
      //<
private:
   void testing(const std::string&);
      //>
      //<

   std::ostream& header();
      //> Header for the unit test case report entry.
      //<
   
   int steps();
      //>
      //<

   std::ostream& footer();
      //> Footer for the unit test case report entry.
      //<

   int theArgc;
      //>
      //<

   char** theArgv;
      //>
      //<

   std::string theCaseName;
      //>
      //<

   std::string theCaseID;
      //>
      //<

   std::string theCaseDescription;
      //>
      //<
   
   std::ostream& theReportStream;
      //>
      //<

   long theDetailsIndent;
      //>
      //<

   TimeT theStartTime;
      //>
      //<

   std::vector<TestStep*> theSteps;
      //>
      //<

   int theStepCount;
      //>
      //<

   FormatFilter theStreamFormatter;
      //>
      //<
};

}

#endif
