% NbrSpAnalyze   Point Cloud Spatial Analysis
%
% File: spanalize.m
%
% Description:
%    Perform local feature attribution via eigenspace analysis.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [classes] = spanalyse(queries, database, ...)
%
% Inputs:
%    queries  - .
%    database - .
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    classes - Structure containing the spatial analysis classifications.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-JAN-28   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Change output to a struct for easier referencing.
%

%******************************************************************************
% spanalyze
%******************************************************************************
function [ classes ] = nbrspanalyze( points, neighborhoods, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    queries  - .
   %
   %    database - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %

   maketimebar = false;
   
   if (~isempty(varargin) && strcmp('timebar',varargin{1}))
      maketimebar = true;
   end
   
   % Sizing metrics
   dimensions=size(points, 1);
   elements=size(neighborhoods, 2);

   % Pre-allocate classes
   norms=zeros(dimensions, elements);
   feats=zeros(dimensions, elements);
   biases=zeros(1,elements);
   ints=zeros(1,elements);

   % This process may take a while. Display time bar while processing.
   if (maketimebar)
      msg='Analyzing Neighborhoods...';
      tstart=tic;
      h = timebar(1, elements, msg, tstart);
   end
   
   % Step through the queries in chunks.
   step=1000;

   if (maketimebar)
      tic;
   end

   for elem=1:step:elements

      last=min(elem+step-1,elements);

      NN=neighborhoods(elem:last);
      
      cells=cell(1,length(NN));

      for n=1:length(NN)
         cells{n}=points(:,NN{n})';
      end
      
      [covs,bias,inty]=fastcov(cells);

      % Compute the neighborhood covariance eigenvalues.
      [V,D]=par_eig(covs);
      
      for nbr=1:length(NN)

         % skip underconstrained neighborhoods (possible with radius searches)
         if (length(NN{nbr})<5 || any(any(isnan(covs{nbr}))))
            continue;
         end

         [~,index]=min(abs(D{nbr}));
         A=V{nbr};
         norms(:,elem+nbr-1)=abs(A(:,index));
         
         % Option 1: Define features as eigenvalues
         % feats(:,elem+nbr-1)=sort(D{nbr},'descend');
         
         % Option 2: Define features as singular values. Eigenvalues are not
         % guaranteed to be in any order. We want them in descending order.
         %
         % max is to get rid of pesky negative zeros.
         feats(:,elem+nbr-1)=sqrt(max(sort(D{nbr},'descend'),0));

         biases(elem+nbr-1)=bias{nbr};
         ints(elem+nbr-1)=inty{nbr};
      end

      % Update the time bar.
      if (maketimebar && toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (maketimebar && all(ishghandle(h, 'figure')))
      close(h);
   end

   % Compute Normalized Features
   energy=sum(feats,1);
   nfeats=bsxfun(@times,feats',1./energy')';
   nfeats(isnan(nfeats))=1;
   
   % Compute Omnivariance
%   omnivar=prod(feats,1).^(1/dimensions);
   omnivar=prod(feats.*feats,1).^(1/dimensions);
%   omnivar=prod(nfeats,1).^(1/dimensions);
   
   % Compute Eigen Entropy
   entropy=-sum(nfeats.*log(nfeats))./log(dimensions);
   entropy(entropy==1)=0;
   
   % Compute Eigen Structure
   structure=1-entropy;
   
   % Compute Fractional Anisotropy
   evmeans=mean(feats,1);
   evdevs=feats-repmat(evmeans,dimensions,1);
   numer=dimensions.*sum(evdevs.^2,1);
   denom=(dimensions-1).*sum(feats.^2,1);
   fa=(numer./denom).^(1/2);
   
   % Compute Anisotropy
   ani=(feats(1,:)-feats(dimensions,:))./(feats(1,:));
   
   % Compute Isotropy
   iso=1-ani;
   
   % Compute dimensional degree
   dims=zeros(size(feats));
   dims(1:dimensions-1,:)=(feats(1:dimensions-1,:)-feats(2:dimensions,:))./...
                           repmat(feats(1,:),dimensions-1,1);
   dims(dimensions,:)=iso;
   
   % Compute Dimensional Embedding
   [~,label]=max(dims,[],1);

   % Compute dimensional entropy
%   alpha=[dims;iso];
   alpha=dims;
   de=-sum(alpha.*log(alpha))./log(dimensions);
   
   if (~isreal(de))
      disp('de...');
      disp(de);
      disp('feats...');
      disp(feats);
      disp('eigs...');
      disp(feats.*feats);
   end
   
   % Populate feature classes
   classes.features=feats;
   classes.normals=norms;
   classes.dimensionality=dims;
   classes.isotropy=iso;
   classes.anisotropy=ani;
   classes.FA=fa;
   classes.entropy=entropy;
   classes.structure=structure;
   classes.omnivariance=omnivar;
   classes.labeling=label;
   classes.biases=biases;
   classes.intensity=ints;
   classes.de=de;

end
