% SpAnalyze   Point Cloud Spatial Analysis
%
% File: spanalize.m
%
% Description:
%    Perform local feature attribution via eigenspace analysis.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [classes] = spanalyse(queries, database, ...)
%
% Inputs:
%    queries  - .
%    database - .
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    classes - Structure containing the spatial analysis classifications.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-JAN-28   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Change output to a struct for easier referencing.
%

%******************************************************************************
% spanalyze
%******************************************************************************
function [ classes ] = spanalyze( queries, points, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    queries  - .
   %
   %    database - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %
   
   userParams = parseInputs(varargin{:}); % Parse Inputs
   
   % Sizing metrics
   dimensions=size(queries, 1);
   elements=size(queries, 2);

   % Pre-allocate classes
   norms=zeros(size(queries));
   feats=zeros(size(queries));
   biases=zeros(1,elements);
   ints=zeros(1,elements);

   % Build database
   database=VpTree(points);
   
   % This process may take a while. Display time bar while processing.
   msg='Computing Structure Tensors...';
   tstart=tic;
   h = timebar(1, elements, msg, tstart);
   
   % Step through the queries in chunks.
   step=1000;

   tic;

   for elem=1:step:elements

      % The last available element may be closer than elem + step.
      last=min(elem+step-1,elements);
      
      % Get the nearest neighbors of elem
      if (userParams.radius > 0 && userParams.neighbors <= 0)
         % Perform a fixed radius search
         NN=database.rnn(queries(:,elem:last),...
                         userParams.radius);
      elseif (userParams.radius <= 0 && userParams.neighbors > 0)
         % Search unconstrained neighbors
         NN=database.knn(queries(:,elem:last),...
                         userParams.neighbors);
      elseif (userParams.radius > 0 && userParams.neighbors > 0)
         % Search constrained to radius
         NN=database.knn(queries(:,elem:last),...
                         userParams.neighbors,...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.counts) && userParams.radius <= 0)
         % Search unconstrained neighbors
         NN=database.knn(queries(:,elem:last),...
                         max(userParams.counts));
      elseif (~isempty(userParams.counts) && userParams.radius > 0)
         % Search constrained to radius
         NN=database.knn(queries(:,elem:last),...
                         max(userParams.counts),...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.steps))
         % Perform a fixed radius search
         [NN,DISTS]=database.rnn(queries(:,elem:last),...
                                 max(userParams.steps));
      end
      
%       % Compute neighborhood covariance, intensity and bias values.
%       if (isempty(userParams.counts) && isempty(userParams.steps))
         % Not using adaptive neighborhoods.
         cells=cell(1,length(NN));
         
         for n=1:length(NN)
            cells{n}=points(:,NN{n})';
         end
         
         [covs,bias,inty]=fastcov(cells);
      
%       elseif (~isempty(userParams.counts))
%          
%          % Using variable neighbor count based adaptive neighborhoods.
% 
%          % TODO: Conoslidate the common functionality in adaptive methods into
%          % a single function.
% 
%          tempfeats=zeros(dimensions,size(NN,1));
%          tempde=zeros(length(userParams.counts),size(NN,1));
%          
%          tempcovs=cell(length(userParams.counts),1);
%          tempbias=cell(length(userParams.counts),1);
%          tempinty=cell(length(userParams.counts),1);
% 
%          % Compute candidiate values (HACKY)
%          for c=1:length(userParams.counts)
%             cells=cell(1,length(NN));
%             
%             for n=1:length(NN)
%                cells{n}=points(:,NN{n}(1:userParams.counts(c)))';
%             end
%             
%             [tempcovs{c},tempbias{c},tempinty{c}]=fastcov(cells);
%             
%             for nbr=1:size(NN,1)
%                [~,D]=par_eig(tempcovs{c}{nbr});
%                tempfeats(:,nbr)=sort(diag(D),'descend');
%             end
%             
%             % Compute dimensional degree
%             % Do not use the following definition based on singular values:
%             % tempdims=(sqrt(tempfeats(1:dimensions-1,:))-...
%             %           sqrt(tempfeats(2:dimensions,:)))./...
%             %           repmat(sqrt(tempfeats(1,:)),dimensions-1,1);
%             tempdims=(tempfeats(1:dimensions-1,:)-...
%                       tempfeats(2:dimensions,:))./...
%                       repmat(tempfeats(1,:),dimensions-1,1);
%                    
%             % Compute isotropy
%             tempiso=(tempfeats(dimensions,:))./(tempfeats(1,:));
% 
%             % Compute dimensional entropy
%             tempalpha=[tempdims;tempiso];
%             tempde(c,:)=-sum(tempalpha.*log(tempalpha))./log(dimensions);
%          end
%          
%          % Minimize on dimensional entropy (HACKY)
%          [~,ind]=min(tempde);
%          
%          covs=tempcovs{ind};
%          bias=tempbias{ind};
%          inty=tempinty{ind};
%       elseif (~isempty(userParams.steps))
%       
%          % Using variable neighborhood radii based adaptive neighborhoods.
%          
%          % TODO: Conoslidate the common functionality in adaptive methods into
%          % a single function.
%          
%          tempfeats=zeros(dimensions,size(NN,1));
%          tempde=zeros(length(userParams.steps),size(NN,1));
%          
%          tempcovs=cell(length(userParams.steps),1);
%          tempbias=cell(length(userParams.steps),1);
%          tempinty=cell(length(userParams.steps),1);
% 
%          % Compute candidiate values (HACKY)
%          for c=1:length(userParams.steps)
%             
%             nbrs=cell(1,length(DISTS));
%             
%             for n=1:length(DISTS)
%                nbrs{n}=sum(DISTS{n} <= userParams.steps(c))';
%             end
%             
%             cells=cell(1,length(NN));
%             
%             for n=1:length(NN)
%                cells{n}=points(:,NN{n}(DISTS{n} <= userParams.steps(c)))';
%             end
%             
%             [tempcovs{c},tempbias{c},tempinty{c}]=fastcov(cells);
%             
%             for nbr=1:size(NN,1)
%                if (nbrs{nbr}<5)
%                   continue;
%                end
%                
%                [~,D]=par_eig(tempcovs{c}{nbr});
%                tempfeats(:,nbr)=sort(diag(D),'descend');
%             end
%             
%             % Compute dimensional degree
%             % Do not use the following definition based on singular values:
%             % tempdims=(sqrt(tempfeats(1:dimensions-1,:))-...
%             %           sqrt(tempfeats(2:dimensions,:)))./...
%             %           repmat(sqrt(tempfeats(1,:)),dimensions-1,1);
%             tempdims=(tempfeats(1:dimensions-1,:)-...
%                       tempfeats(2:dimensions,:))./...
%                       repmat(tempfeats(1,:),dimensions-1,1);
%                    
%             % Compute isotropy
%             tempiso=(tempfeats(dimensions,:))./(tempfeats(1,:));
%             
%             % Compute dimensional entropy
%             tempalpha=[tempdims;tempiso];
%             tempde(c,:)=-sum(tempalpha.*log(tempalpha))./log(dimensions);
%          end
%          
%          % Minimize on dimensional entropy (HACKY)
%          [~,ind]=nanmin(tempde);
%          
%          covs=tempcovs{ind};
%          bias=tempbias{ind};
%          inty=tempinty{ind};
%       else
%          error('Unhandled Condition');
%       end

      % Compute the neighborhood covariance eigenvalues.
      [V,D]=par_eig(covs);
      
      for nbr=1:length(NN)

         % skip underconstrained neighborhoods (possible with radius searches)
         if (userParams.radius > 0 && (length(NN{nbr})<5 ||...
             any(any(isnan(covs{nbr})))))
            continue;
         end

         [~,index]=min(abs(D{nbr}));
         A=V{nbr};
         norms(:,elem+nbr-1)=abs(A(:,index));
         
         % Option 1: Define features as eigenvalues
         % feats(:,elem+nbr-1)=sort(D{nbr},'descend');
         
         % Option 2: Define features as singular values. Eigenvalues are not
         % guaranteed to be in any order. We want them in descending order.
         feats(:,elem+nbr-1)=sqrt(sort(D{nbr},'descend'));

         biases(elem+nbr-1)=bias{nbr};
         ints(elem+nbr-1)=inty{nbr};
      end

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end

   % Compute Normalized Features
   energy=sum(feats,1);
   nfeats=bsxfun(@times,feats',1./energy')';
   nfeats(isnan(nfeats))=1;
   
   % Compute Omnivariance
%   omnivar=prod(feats,1).^(1/dimensions);
   omnivar=prod(feats.*feats,1).^(1/dimensions);
%   omnivar=prod(nfeats,1).^(1/dimensions);
   
   % Compute Eigen Entropy
   entropy=-sum(nfeats.*log(nfeats))./log(dimensions);
   entropy(entropy==1)=0;
   
   % Compute Eigen Structure
   structure=1-entropy;
   
   % Compute Fractional Anisotropy
   evmeans=mean(feats,1);
   evdevs=feats-repmat(evmeans,dimensions,1);
   numer=dimensions.*sum(evdevs.^2,1);
   denom=(dimensions-1).*sum(feats.^2,1);
   fa=(numer./denom).^(1/2);
   
   % Compute Anisotropy
   ani=(feats(1,:)-feats(dimensions,:))./(feats(1,:));
   
   % Compute Isotropy
   iso=1-ani;
   
   % Compute dimensional degree
   dims=zeros(size(feats));
   dims(1:dimensions-1,:)=(feats(1:dimensions-1,:)-feats(2:dimensions,:))./...
                           repmat(feats(1,:),dimensions-1,1);
   dims(dimensions,:)=iso;
   
   % Compute Dimensional Embedding
   [~,label]=max(dims,[],1);

   % Compute dimensional entropy
%   alpha=[dims;iso];
   alpha=dims;
   de=-sum(alpha.*log(alpha))./log(dimensions);
   
   % Populate feature classes
   classes.features=feats;
   classes.normals=norms;
   classes.dimensionality=dims;
   classes.isotropy=iso;
   classes.anisotropy=ani;
   classes.FA=fa;
   classes.entropy=entropy;
   classes.structure=structure;
   classes.omnivariance=omnivar;
   classes.labeling=label;
   classes.biases=biases;
   classes.intensity=ints;
   classes.de=de;

% TODO: Add curvature analysis back into spatial analyzer. Will require
% re-work.
%
%    st=zeros(size(QUERIES));
% 
%    msg='Computing Normal Structure Tensors...';
%    tstart=tic;
%    h = timebar(1, elements, msg, tstart);
% 
%    for elem=1:step:elements
%     
%       last=min(elem+step-1,elements);
%     
%       % Get the nearest neighbors of elem
%       if (userParams.neighbors <= 0)
%           % Perform a fixed radius search
%           NN=DATABASE.rnn(QUERIES(:,elem:last),...
%                           userParams.radius);
%       else
%           if (userParams.radius <= 0)
%               % Search unconstrained
%               NN=DATABASE.knn(QUERIES(:,elem:last),...
%                               userParams.neighbors);
%           else
%               % Search constrained to radius
%               NN=DATABASE.knn(QUERIES(:,elem:last),...
%                               userParams.neighbors,...
%                               'lim',userParams.radius);
%           end
%       end
% 
%       for nbr=1:size(NN,1)
%          normals=norms(:,NN{nbr});
% 
%          S=svd(normals',0);
%          st(:,elem+nbr-1)=S.*S./(length(NN{nbr})-1);
% 
% %          D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %          st(:,elem+nbr-1)=sort(D,'descend');
% 
% %          ST=normals*normals';
% 
% %          if (nbr == 1)
% %              D=eig(1/(length(NN{nbr})-1)*(normals*normals'));
% %              disp('EIG:');
% %              disp(sqrt(D));
% %           
% %              S=svd(normals',0);
% %              S = S ./ sqrt(length(NN{nbr})-1);
% %              disp('SVD:');
% %              disp(abs(S));
% % 
% %          end
% 
%       end
% 
%       if(toc > 1)
%          tic;
%          h = timebar(elem, elements, msg, tstart, h);
%       end
%    end
% 
%    if (all(ishghandle(h, 'figure')))
%       close(h);
%    end

end

%******************************************************************************
% parseInputs
%******************************************************************************
function [userParams] = parseInputs(varargin)
   % Support function to parse inputs into userParams structure
   %
   % Parameters:
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    userParams - .

   userParams = struct('radius', 0, 'neighbors', 0, 'counts', [], 'steps', []);
   
   if isempty(varargin)
      error('Damkjer:InvalidOptions', ...
            ['A neighborhood size must be specified, either directly or '...
             'via optional parameters.']);
   end
   
   if length(varargin) == 1 || ~isnumeric(varargin{2})
      value = varargin{1};
      
      if (isscalar(value)  && ...
          isreal(value)    && ...
          value >= 5)
         userParams.neighbors = fix(value);
      else
         error('Damkjer:InvalidCount', ...
               ['Number of Neighbors must be a real valued positive '...
                'integer greater or equal to 5: ' num2str(value)]);
      end
      
      varargin(1) = [];
   end
   
   % Parse the Property/Value pairs
   if rem(length(varargin), 2) ~= 0
      error('Damkjer:PropertyValueNotPair', ...
            ['Additional arguments must take the form of Property/Value '...
             'pairs']);
   end
   
   propertyNames = {'neighbors', 'radius', 'counts', 'steps'};

   while ~isempty(varargin)
      property = varargin{1};
      value    = varargin{2};
      
      % If the property has been supplied in a shortened form, lengthen it
      iProperty = find(strncmpi(property, propertyNames, length(property)));
      
      if isempty(iProperty)
         error('Damkjer:InvalidProperty', 'Invalid Property');
      elseif length(iProperty) > 1
         error('Damkjer:AmbiguousProperty', ...
               'Supplied shortened property name is ambiguous');
      end
      
      property = propertyNames{iProperty};
      
      switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
          userParams.neighbors = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Number of Neighbors must be a real valued positive '...
                   'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidRadius', ...
                  'Radius must be a real valued positive scalar');
         end
      case 'counts'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value >= 5))
            userParams.counts = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Counts must be a sorted vector of real valued positive '...
                   'integers greater or equal to 5']);
         end
      case 'steps'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value > 0))
            userParams.steps = value;
         else
            error('Damkjer:InvalidSteps', ...
                  ['Steps must be a sorted vector of real valued positive '...
                   'values']);
         end
      end
      
      varargin(1:2) = [];
   end

   % Check for mutually exclusive options.
   if (~isempty(userParams.counts) && userParams.neighbors >= 5)
      error('Damkjer:MutExOpts', ...
            '''neighbors'' and ''counts'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.steps) && userParams.radius > 0)
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''radius'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.counts) && ~isempty(userParams.steps))
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''counts'' options are mutually exclusive');
   end

   % Default, if necessary.
   if (userParams.neighbors <= 0 && ...
       userParams.radius <= 0 && ...
       isempty(userParams.counts) && ...
       isempty(userParams.steps))
      userParams.radius = 1;
   end
end
