% Thin_Dyn   Point Cloud Thinning (Dyn et al)
%
% File: thin_Dyn.m
%
% Description:
%    Perform thinning according to Dyn 2004.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [results] = spanalyse(points, threshold, ...)
%
% Inputs:
%    points - .
%    threshold - must be positive. between 0-1, treated as percentage, > 1
%                points.
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    results - Thinned point cloud.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    Dyn reference.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-DEC-28  Initial coding.
%

%******************************************************************************
% thin_Dyn
%******************************************************************************
function [ results ] = thin_Damkjer( points, threshold, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    points  - .
   %
   %    threshold - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %
   
   userParams = parseInputs(varargin{:}); % Parse Inputs

   % Make sure we have unique points... we barf otherwise...
   disp('Culling duplicate points...');
   tstart=tic;
   points = unique(points.','rows').';
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   disp('Indexing points...');
   tstart=tic;
   database = VpTree(points);
   queries = points;
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   database.excludeWithin(0.001); % Set mm precision on searches.
   
   % Sizing metrics
   dimensions=size(queries, 1);
   elements=size(queries, 2);
   
   nbrs  = cell(1,elements);
   
   %***
   % CREATE NEIGHBORHOODS
   %***

   % This process may take a while. Display time bar while processing.
   msg='Creating Neigborhoods...';
   tstart=tic;
   h = timebar(1, elements, msg, tstart);

   disp(msg);

   % Step through the queries in chunks.
   step=1000;

   tic;

   for elem=1:step:elements

      % The last available element may be closer than elem + step.
      last=min(elem+step-1,elements);
      
      % Get the nearest neighbors of elem
      if (userParams.radius > 0 && userParams.neighbors <= 0)
         % Perform a fixed radius search
         nbrs(elem:last)=database.rnn(queries(:,elem:last),...
                         userParams.radius);
      elseif (userParams.radius <= 0 && userParams.neighbors > 0)
         % Search unconstrained neighbors
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         userParams.neighbors);
      elseif (userParams.radius > 0 && userParams.neighbors > 0)
         % Search constrained to radius
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         userParams.neighbors,...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.counts) && userParams.radius <= 0)
         % Search unconstrained neighbors
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         max(userParams.counts));
      elseif (~isempty(userParams.counts) && userParams.radius > 0)
         % Search constrained to radius
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         max(userParams.counts),...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.steps))
         % Perform a fixed radius search
         [nbrs(elem:last),DISTS]=database.rnn(queries(:,elem:last),...
                                 max(userParams.steps));
      end

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end

   disp(['...done in ' num2str(toc(tstart)) 's']);

%% Obsolete code   
   %***
   % COMPUTE DUAL NEIGHBORHOODS (THIS IS SLOW. LET'S MEX IT)
   %***

%    % This process may take a while. Display time bar while processing.
%    msg='Computing Dual Neigborhoods...';
%    tstart=tic;
%    h = timebar(1, elements, msg, tstart);
%    
%    disp(msg);
% 
%    tic;
%    for elem=1:elements
%       
%       % Build up dual set
%       for nbr=1:length(nbrs{elem})
%          n = nbrs{elem}(nbr);
%          duals{n}=[duals{n}(:); elem];
%       end
% 
%       % Update the time bar.
%       if (toc > 1)
%          tic;
%          h = timebar(elem, elements, msg, tstart, h);
%       end
%    end
% 
%    % Close the time bar, if still open.
%    if (all(ishghandle(h, 'figure')))
%       close(h);
%    end
% 
%    disp(['...done in ' num2str(toc(tstart)) 's']);
%%
   %***
   % COMPUTE DUAL NEIGHBORHOODS -- MEX'D
   %***

   disp('Computing Dual Neigborhoods...');
   tstart=tic;
   duals=fastsetdual(nbrs);
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   %***
   % COMPUTE SIGNIFICANCES
   %***
   disp('Computing Significances...');
   tstart=tic;

   feats = nbrspanalyze(points, nbrs, 'timebar');

   disp(['...done in ' num2str(toc(tstart)) 's']);

   sigs = feats.de;
%   sigs = sqrt(feats.biases.*feats.de);
%   sigs = 0.5 * (feats.biases + feats.de);

   %***
   % PARTITION THE DATA SET
   %***

   disp('Partition by Label...');
   tstart=tic;

   parts = cell(1,dimensions);

   for dim=1:dimensions
      parts{dim}=uint64(find(feats.labeling==dim));
   end

   [~,idx]=sort(cellfun('length',parts),'ascend');
   parts=parts(idx);
   
   disp(['...done in ' num2str(toc(tstart)) 's']);

   %***
   % THIN BY LABEL
   %***

   fraction=0;

   removed=zeros(1,elements-threshold);
   dual_size=zeros(1,elements-threshold);
   current=1;

   removed_wgts=cell(1,3);
   fellback = 0;

   for dim=1:dimensions
      desired  = threshold * length(parts{dim}) / elements + fraction;
      tau      = floor(desired);
      fraction = desired-tau;
      heap     = SplayTree(sigs(parts{dim}),parts{dim});
      n        = length(parts{dim})-tau;
      
      removed_wgts{dim}=zeros(1,n);

%      tau=threshold;
%      heap = SplayTree(feats.de);
%      n = length(feats.de) - tau;
%      removed_wgts=zeros(1,n);

      msg=['Thin Dimension ' num2str(dim) '(' num2str(length(parts{dim})) ...
                                          '->' num2str(tau) ')...'];
%      msg='Thin...';
      tstart=tic;
      h = timebar(1, n, msg, tstart);

      disp(msg);
      tic;

      for x=1:n
%      if false

         % Remove Element
         [wgt, idx]=heap.pop_head();

         removed(current)=idx;
         removed_wgts{dim}(x)=wgt;

         % Find neighborhoods that contain removed element
         neighborhoods=sort(duals{idx}, 'ascend');
         dual_size(current)=length(neighborhoods);

         % The removed element neighborhood is eliminated, so it no longer
         % contains the neighbors. Update the neighbor duals appropriately.
         neighbors=nbrs{idx};

         for nbr=1:length(neighbors)
            duals{neighbors(nbr)}=duals{neighbors(nbr)}(duals{neighbors(nbr)}~=idx);
         end
         
         % The fallback neighborhood is identical for all neighborhoods, but
         % may be expensive to compute. Allow it to be lazily instantiated.
         fallback_pool = [];
         
         % Replace removed element in each neighborhood with nearest neighbor
         for hood=1:length(neighborhoods)

            % Build up the list of potential new neighbors
%            disp('building candidates...');

%% Obsolete code
%            nbr_pool=unique(vertcat(nbrs{nbrs{neighborhoods(hood)}}));
%            nbr_pool=setdiff(nbr_pool, idx);
%            nbr_pool=setdiff(nbr_pool, nbrs{neighborhoods(hood)});
%%
            nbr_pool=fastsetunion({nbrs{nbrs{neighborhoods(hood)}}});
            non_nbrs=fastsetunion({nbrs{neighborhoods(hood)},idx,neighborhoods(hood)});
            nbr_pool=nbr_pool(~ismembc(nbr_pool(:), non_nbrs(:)));

            if (isempty(nbr_pool))
               if (isempty(fallback_pool))
                  fallback_pool = uint64(find(~ismember(1:size(points,2),removed)));
               end
               fellback = fellback + 1;
               nbr_pool = fallback_pool;
               nbr_pool=nbr_pool(~ismembc(nbr_pool(:), non_nbrs(:)));
            end
            

            % Find the closest
%            disp('finding closest...');

%% Obsolete code
%            % Don't use VP Tree for single nearest neighbor... overhead not
%            % worth it.
%            locdb = VpTree(points(:,nbr_pool));
%            nn=locdb.knn(points(:,idx),1);
%            new_idx=nn{1}
%%
            [~,new_idx]=min(sum((bsxfun(@minus, points(:,nbr_pool), ...
                                        points(:,idx))).^2, 1));
                                                 
            % Update the neighborhood
%            disp('updating neighborhood...'); 

            if (length(nbr_pool(new_idx)) ~= length(nbrs{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx)))
               neighborhoods
               size(fallback_pool)
               non_nbrs
               nbr_pool
               nbr_pool(new_idx)
               
               nbrs{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx)
               
               error('What happened?');
            end
            
            nbrs{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx) = ...
               nbr_pool(new_idx);

            % Update the dual
%            disp('updating dual...');
            duals{nbr_pool(new_idx)}(end + 1,1) = neighborhoods(hood);
            
%             disp(nbrs{neighborhoods(hood)});
%             disp(duals{nbr_pool(new_idx)});
         end
         
         duals{idx}=[];

         % Update analysis for elements
%         disp('updating analysis...');
         locfeats = nbrspanalyze(points, nbrs(neighborhoods));

         locsigs = locfeats.de;
%         locsigs = sqrt(locfeats.biases.*locfeats.de);
%         locsigs = 0.5 * (locfeats.biases + locfeats.de);

         % Update heap, where required
%         size(neighborhoods')
%
%         if (size(sigs(neighborhoods)) ~= size(neighborhoods'))
%            sigsize=size(sigs(neighborhoods))
%            nbrsize=size(neighborhoods')
%            error('Stop');
%         end
         
%         sigs(neighborhoods)
%         neighborhoods'
%         locsigs

         updates=parts{dim}(ismember(parts{dim},neighborhoods));
         sig_upd=locsigs(ismember(neighborhoods,updates));

         if (length(updates) ~= length(sig_upd))
            fellback
            neighborhoods
            updates
            sig_upd
            error('OK, seriously!');
         end
         
         heap.erase(sigs(updates),updates);
         heap.insert(sig_upd,updates);
         sigs(neighborhoods)=locsigs;
         feats.de(neighborhoods)=locfeats.de;
         feats.biases(neighborhoods)=locfeats.biases;
         
         current = current + 1;
         
         % Update the time bar.
         if (toc > 1)
            tic;
            h = timebar(x, n, msg, tstart, h);
         end
      end
      
      % Close the time bar, if still open.
      if (all(ishghandle(h, 'figure')))
         close(h);
      end
      
      disp(['...done in ' num2str(toc(tstart)) 's']);   
   end

   disp(['Generated fallback neighborhood pool ' num2str(fellback) ' time(s).']);
   
   x_max=max(cellfun(@length, removed_wgts));
   y=nan(length(removed_wgts),x_max);
   
   for dim=1:dimensions
      y(dim,1:length(removed_wgts{dim}))=removed_wgts{dim};
   end

%   x_max=length(removed_wgts);
%   y = removed_wgts;

   figure, plot(1:x_max,y);
%   figure, semilogx(1:x_max,y);

   figure, plot(1:length(dual_size),dual_size);

   results = points(:,~ismember(1:size(points,2),removed));

end

%******************************************************************************
% parseInputs
%******************************************************************************
function [userParams] = parseInputs(varargin)
   % Support function to parse inputs into userParams structure
   %
   % Parameters:
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    userParams - .

   userParams = struct('radius', 0, 'neighbors', 0, 'counts', [], 'steps', []);
   
   if isempty(varargin)
      error('Damkjer:InvalidOptions', ...
            ['A neighborhood size must be specified, either directly or '...
             'via optional parameters.']);
   end
   
   if length(varargin) == 1 || ~isnumeric(varargin{2})
      value = varargin{1};
      
      if (isscalar(value)  && ...
          isreal(value)    && ...
          value >= 5)
         userParams.neighbors = fix(value);
      else
         error('Damkjer:InvalidCount', ...
               ['Number of Neighbors must be a real valued positive '...
                'integer greater or equal to 5: ' num2str(value)]);
      end
      
      varargin(1) = [];
   end
   
   % Parse the Property/Value pairs
   if rem(length(varargin), 2) ~= 0
      error('Damkjer:PropertyValueNotPair', ...
            ['Additional arguments must take the form of Property/Value '...
             'pairs']);
   end
   
   propertyNames = {'neighbors', 'radius', 'counts', 'steps'};

   while ~isempty(varargin)
      property = varargin{1};
      value    = varargin{2};
      
      % If the property has been supplied in a shortened form, lengthen it
      iProperty = find(strncmpi(property, propertyNames, length(property)));
      
      if isempty(iProperty)
         error('Damkjer:InvalidProperty', 'Invalid Property');
      elseif length(iProperty) > 1
         error('Damkjer:AmbiguousProperty', ...
               'Supplied shortened property name is ambiguous');
      end
      
      property = propertyNames{iProperty};
      
      switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
          userParams.neighbors = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Number of Neighbors must be a real valued positive '...
                   'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidRadius', ...
                  'Radius must be a real valued positive scalar');
         end
      case 'counts'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value >= 5))
            userParams.counts = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Counts must be a sorted vector of real valued positive '...
                   'integers greater or equal to 5']);
         end
      case 'steps'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value > 0))
            userParams.steps = value;
         else
            error('Damkjer:InvalidSteps', ...
                  ['Steps must be a sorted vector of real valued positive '...
                   'values']);
         end
      end
      
      varargin(1:2) = [];
   end

   % Check for mutually exclusive options.
   if (~isempty(userParams.counts) && userParams.neighbors >= 5)
      error('Damkjer:MutExOpts', ...
            '''neighbors'' and ''counts'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.steps) && userParams.radius > 0)
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''radius'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.counts) && ~isempty(userParams.steps))
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''counts'' options are mutually exclusive');
   end

   % Default, if necessary.
   if (userParams.neighbors <= 0 && ...
       userParams.radius <= 0 && ...
       isempty(userParams.counts) && ...
       isempty(userParams.steps))
      userParams.radius = 1;
   end
end
