% Thin_Dyn   Point Cloud Thinning (Dyn et al)
%
% File: thin_Dyn.m
%
% Description:
%    Perform thinning according to Dyn 2004.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [results] = spanalyse(points, threshold, ...)
%
% Inputs:
%    points - .
%    threshold - must be positive. between 0-1, treated as percentage, > 1
%                points.
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    results - Thinned point cloud.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    Dyn reference.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-DEC-28  Initial coding.
%

%******************************************************************************
% thin_Dyn
%******************************************************************************
function [ results ] = thin_Dyn( points, threshold, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    points  - .
   %
   %    threshold - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %
   
   userParams = parseInputs(varargin{:}); % Parse Inputs

   % Make sure we have unique points... we barf otherwise...
   disp('Culling duplicate points...');
   tstart=tic;
   points = unique(points.','rows').';
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   disp('Indexing points...');
   database = VpTree(points);
   queries = points;
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   database.excludeWithin(0.001); % Set mm precision on searches.
   
   % Sizing metrics
   dimensions=size(queries, 1);
   elements=size(queries, 2);

   % Pre-allocate classes
%   norms=zeros(size(queries));
%   feats=zeros(size(queries));
%   biases=zeros(1,elements);
%   ints=zeros(1,elements);


   nbrs  = cell(1,elements);
   duals = cell(1,elements);
   tests = cell(1,elements);
   
   sigs=zeros(1,elements);
   
   for elem=1:elements
      tests{elem}=elem;
   end

   %***
   % COMPUTE NEIGHBORHOODS
   %***

   % This process may take a while. Display time bar while processing.
   msg='Computing Neigborhoods...';
   tstart=tic;
   h = timebar(1, elements, msg, tstart);
   
   % Step through the queries in chunks.
   step=1000;

   tic;

   for elem=1:step:elements

      % The last available element may be closer than elem + step.
      last=min(elem+step-1,elements);
      
      % Get the nearest neighbors of elem
      if (userParams.radius > 0 && userParams.neighbors <= 0)
         % Perform a fixed radius search
         nbrs(elem:last)=database.rnn(queries(:,elem:last),...
                         userParams.radius);
      elseif (userParams.radius <= 0 && userParams.neighbors > 0)
         % Search unconstrained neighbors
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         userParams.neighbors);
      elseif (userParams.radius > 0 && userParams.neighbors > 0)
         % Search constrained to radius
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         userParams.neighbors,...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.counts) && userParams.radius <= 0)
         % Search unconstrained neighbors
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         max(userParams.counts));
      elseif (~isempty(userParams.counts) && userParams.radius > 0)
         % Search constrained to radius
         nbrs(elem:last)=database.knn(queries(:,elem:last),...
                         max(userParams.counts),...
                         'lim',userParams.radius);
      elseif (~isempty(userParams.steps))
         % Perform a fixed radius search
         [nbrs(elem:last),DISTS]=database.rnn(queries(:,elem:last),...
                                 max(userParams.steps));
      end

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end

   %***
   % COMPUTE SIGNIFICANCES (THIS IS SLOW AND SHOULD BE EASY TO PARALLELIZE)
   %***

   % This process may take a while. Display time bar while processing.
   msg='Computing Significances...';
   tstart=tic;
   h = timebar(1, elements, msg, tstart);
   
   % Step through the queries in chunks.
   step=1000;

   tic;

   for elem=1:step:elements

      % The last available element may be closer than elem + step.
      last=min(elem+step-1,elements);

      % Compute significance
      sigs(elem:last)=sig_Dyn(points,tests(elem:last),nbrs(elem:last));

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end
   
   results = [points;sigs];
   
%    %***
%    % COMPUTE DUAL NEIGHBORHOODS (THIS IS SLOW AND SHOULD BE EASY TO PARALLELIZE!)
%    %***
%    
%    % This process may take a while. Display time bar while processing.
%    msg='Computing Dual Neigborhoods...';
%    tstart=tic;
%    h = timebar(1, elements, msg, tstart);
%    
%    tic;
% 
%    for elem=1:elements
%       % Build up dual set
%       for nbr=1:length(nbrs{elem})
%          n = nbrs{elem}(nbr);
%          duals{n}=[duals{n}(:); elem];
%       end
% 
% %         if rem(elem-1,100000)==0
% %            for insp=1:10
% %            disp(insp);
% %            disp(duals{insp}');
% %            end
% %         end
%          
% %      end
%       
%       % Update the time bar.
%       if (toc > 1)
%          tic;
%          h = timebar(elem, elements, msg, tstart, h);
%       end
%    end
%    
%    % Close the time bar, if still open.
%    if (all(ishghandle(h, 'figure')))
%       close(h);
%    end
% 
%    results = points;
end

%******************************************************************************
% parseInputs
%******************************************************************************
function [userParams] = parseInputs(varargin)
   % Support function to parse inputs into userParams structure
   %
   % Parameters:
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    userParams - .

   userParams = struct('radius', 0, 'neighbors', 0, 'counts', [], 'steps', []);
   
   if isempty(varargin)
      error('Damkjer:InvalidOptions', ...
            ['A neighborhood size must be specified, either directly or '...
             'via optional parameters.']);
   end
   
   if length(varargin) == 1 || ~isnumeric(varargin{2})
      value = varargin{1};
      
      if (isscalar(value)  && ...
          isreal(value)    && ...
          value >= 5)
         userParams.neighbors = fix(value);
      else
         error('Damkjer:InvalidCount', ...
               ['Number of Neighbors must be a real valued positive '...
                'integer greater or equal to 5: ' num2str(value)]);
      end
      
      varargin(1) = [];
   end
   
   % Parse the Property/Value pairs
   if rem(length(varargin), 2) ~= 0
      error('Damkjer:PropertyValueNotPair', ...
            ['Additional arguments must take the form of Property/Value '...
             'pairs']);
   end
   
   propertyNames = {'neighbors', 'radius', 'counts', 'steps'};

   while ~isempty(varargin)
      property = varargin{1};
      value    = varargin{2};
      
      % If the property has been supplied in a shortened form, lengthen it
      iProperty = find(strncmpi(property, propertyNames, length(property)));
      
      if isempty(iProperty)
         error('Damkjer:InvalidProperty', 'Invalid Property');
      elseif length(iProperty) > 1
         error('Damkjer:AmbiguousProperty', ...
               'Supplied shortened property name is ambiguous');
      end
      
      property = propertyNames{iProperty};
      
      switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
          userParams.neighbors = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Number of Neighbors must be a real valued positive '...
                   'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidRadius', ...
                  'Radius must be a real valued positive scalar');
         end
      case 'counts'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value >= 5))
            userParams.counts = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Counts must be a sorted vector of real valued positive '...
                   'integers greater or equal to 5']);
         end
      case 'steps'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value > 0))
            userParams.steps = value;
         else
            error('Damkjer:InvalidSteps', ...
                  ['Steps must be a sorted vector of real valued positive '...
                   'values']);
         end
      end
      
      varargin(1:2) = [];
   end

   % Check for mutually exclusive options.
   if (~isempty(userParams.counts) && userParams.neighbors >= 5)
      error('Damkjer:MutExOpts', ...
            '''neighbors'' and ''counts'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.steps) && userParams.radius > 0)
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''radius'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.counts) && ~isempty(userParams.steps))
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''counts'' options are mutually exclusive');
   end

   % Default, if necessary.
   if (userParams.neighbors <= 0 && ...
       userParams.radius <= 0 && ...
       isempty(userParams.counts) && ...
       isempty(userParams.steps))
      userParams.radius = 1;
   end
end
