//*****************************************************************************
// FILE:        ClassHandle.h
//
//    Copyright (C)  2012 Kristian L. Damkjer.
//
// DESCRIPTION:
//>   The interface definition for C++ class handle objects.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-OCT-08  K. Damkjer
//               Initial Coding.
//  2013-JUN-24  K. Damkjer
//               Clarified Error ID.
//<
//*****************************************************************************
#ifndef Damkjer_ClassHandle_HEADER
#define Damkjer_ClassHandle_HEADER

#include <typeinfo>   // USES the typeid operator
#include <string>     // HASA std::string data member

//***
// Fix "wide char" definition for older versions of MATLAB. This must be placed
// after other includes and before the MathWorks includes.
//***
#if (defined(MATLAB_MAJOR) && defined(MATLAB_MINOR))
   #if MATLAB_MAJOR <= 7 && MATLAB_MINOR <= 10 && defined(_CHAR16T)
      #define CHAR16_T
   #endif
#endif

#include <tmwtypes.h> // USES MathWorks types
#include "matrix.h"   // HASA mxArray interface, USES matrix interfaces
#include "mex.h"      // USES MEX error interfaces, mexLock and mexUnlock

namespace Damkjer
{
//*****************************************************************************
// CLASS: ClassHandle
//>   A C++ object wrapper to enable object persistence across MEX function
//    calls in MATLAB.
//
//    This class implements an encapsulation scheme to wrap an object in a
//    generic handle container that's allocated on the heap. This container is
//    then reinterpreted as a MATLAB memory-managed matrix for safe transport
//    between MEX function calls. The general scheme was developed through
//    collaboration on the MathWorks forums:
//
//    http://www.mathworks.com/matlabcentral/newsreader/view_thread/278243
//
//    @note
//    Default construction, copy construction and assignment are explicitly
//    disallowed to preserve the lifetime guarantees of the pointer.
//    ClassHandle owns the pointer it contains. Thus, when the ClassHandle is
//    destroyed, the pointer is deleted to free the resources for the wrapped
//    object. ClassHandle objects are encouraged to be allocated on the heap
//    via ClassHandle::createHandle by using the named constructor idiom. From
//    a MEX source file, the ClassHandle objects should be destroyed using the
//    ClassHandle::destroyHandle method.
//
//    @tparam ObjectT The wrapped object type.
//<
//*****************************************************************************
template<typename ObjectT>
class ClassHandle
{
public:
   ~ClassHandle();
      //> Destroy the ClassHandle object and perform clean-up activities.
      //<
   
   ObjectT& object() const { return *thePointer; }
      //> Return the object encapsulated by the ClassHandle.
      //
      //  @tparam ObjectT The wrapped object type.
      //  @return         A reference to the base class object.
      //<
   
   bool isValid() const;
      //> Perform a validity check on the ClassHandle object.
      //<

   static mxArray* createHandle(ObjectT*);
      //> Create a MathWorks index handle to the argument.
      //<

   static ObjectT& handleReference(const mxArray*);
      //> Reinterpret a MathWorks index as a C++ object reference.
      //<

   static void destroyHandle(const mxArray*);
      //> Free resources for the handle object represented by the MathWorks
      //  index.
      //<

private:

//   typedef typename ClassHandle<ObjectT> HandleT;
   typedef ClassHandle<ObjectT> HandleT;
      //> Shortcut reference to this class type.
      //<

   ClassHandle();
      //> Explicitly disable the compiler-generated default constructor. No
      //  definition provided.
      //<

   ClassHandle(const ClassHandle<ObjectT>&);
      //> Explicitly disable the compiler-generated copy constructor. No
      //  definition provided.
      //<

   ClassHandle<ObjectT>& operator=(const ClassHandle<ObjectT>&);
      //> Explicitly disable the compiler-generated assignment operator. No
      //  definition provided.
      //
      //  @return The updated left-hand side when not disabled.
      //<

   ClassHandle(ObjectT*);
      //> Initialize a ClassHandle with a pointer to an ObjectT object.
      //<

   ObjectT* thePointer;
      //> The pointer to the encapsulated object.
      //<
   
   std::string theSignature;
      //> The file signature. Used for validation.
      //<
};

//*****************************************************************************
// ClassHandle::~ClassHandle
//>   Destroy the ClassHandle object and perform clean-up activities.
//
//    Free resources for the ClassHandle object and invoke mexUnock to allow
//    calling MEX files to be cleared after memory management tasks have
//    completed.
//
//    @tparam ObjectT The wrapped object type.
//<
//*****************************************************************************
template<typename ObjectT>
inline
ClassHandle<ObjectT>::~ClassHandle()
{
   delete thePointer;
   thePointer = 0;

   mexUnlock();
}

//*****************************************************************************
// ClassHandle::isValid()
//>   Perform a validity check on the ClassHandle object.
//
//    This method returns true if and only if the ClassHandle object passes
//    validity checks. This method prevents the arbitrary reinterpretation of
//    non-ClassWHandle objects.
//
//    @tparam  ObjectT The wrapped object type.
//    @returns         true if and only if the ClassHandle object passes
//                     validity checks.
//<
//*****************************************************************************
template<typename ObjectT>
inline
bool
ClassHandle<ObjectT>::isValid() const
{
   return (theSignature == typeid(ObjectT).name());
}

//*****************************************************************************
// ClassHandle::ClassHandle(ObjectT* const)
//>   Initialize a ClassHandle with a pointer to an ObjectT object.
//
//    This constructor is private to force objects to be created using the
//    ptrAsMat named constructor idiom. This also forces ClassHandles to be
//    allocated on the heap instead of on the stack. This method is responsible
//    for invoking mexLock to prevent MATLAB from arbitrarily clearing memory
//    managed by the wrapper.
//
//    @tparam ObjectT  The wrapped object type.
//    @param  aPointer A pointer to an object to be wrapped with a ClassHandle.
//<
//*****************************************************************************
template<typename ObjectT>
inline
ClassHandle<ObjectT>::ClassHandle(ObjectT* aPointer)
   : thePointer(aPointer)
   , theSignature(typeid(ObjectT).name())
{
   mexLock();
}

}

#if MAKE_DEPENDING || TEMPLATE_INCLUSION
#include "ClassHandle.hpp"
#endif

#endif
