//****************************************************************************
// FILE:        ClassHandle.hpp
//
//    Copyright (C)  2012 Kristian L. Damkjer.
//
// DESCRIPTION:
//>   The template implementation for C++ class handle objects.
//<
//
// LIMITATIONS:
//>   No known limitations.
//<
//
// SOFTWARE HISTORY:
//> 2012-OCT-08  K. Damkjer
//               Initial Coding.
//  2013-JUN-24  K. Damkjer
//               Clarified Error ID.
//<
//*****************************************************************************

#include "ClassHandle.h"

namespace Damkjer
{

//*****************************************************************************
// ClassHandle::createHandle(ObjectT* const)
//>   Create a MathWorks index handle to the argument.
//
//    Safely encapsulate a pointer to a C++ object as a MathWorks matrix for
//    persistence between MEX function calls.
//
//    @note
//    This function is a named constructor for ClassHandle objects.
//
//    @tparam ObjectT  The wrapped object type.
//    @param  aPointer A pointer to an object to be wrapped with a
//                     ClassHandle.
//    @returns         A pointer to a MATLAB matrix object holding the pointer
//                     to the wrapped object handle encoded as an a MathWorks
//                     index value.
//<
//*****************************************************************************
template<typename ObjectT>
mxArray*
ClassHandle<ObjectT>::createHandle(ObjectT* aPointer)
{
   mxArray* aMatrix = mxCreateNumericMatrix(1, 1, mxINDEX_CLASS, mxREAL);

   *(static_cast<mwIndex*>(mxGetData(aMatrix))) =
      reinterpret_cast<mwIndex>(new HandleT(aPointer));

   return aMatrix;
}

//*****************************************************************************
// ClassHandle::object(const mxArray* const)
//>   Reinterpret a MathWorks index as a C++ object reference.
//
//    @tparam ObjectT The wrapped object type.
//    @param  aMatrix A MathWorks matrix containing a single index value to be
//                    reinterpretted as a reference to an object of type
//                    ObjectT.
//    @returns        A reference the wrapped object of type ObjectT.
//<
//*****************************************************************************
template<typename ObjectT>
ObjectT&
ClassHandle<ObjectT>::handleReference(const mxArray* aMatrix)
{
   if (mxGetNumberOfElements(aMatrix) != 1    ||
       mxGetClassID(aMatrix) != mxINDEX_CLASS ||
       mxIsComplex(aMatrix))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsObj:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }

   HandleT* handle =
      reinterpret_cast<HandleT*>(*(static_cast<mwIndex*>(mxGetData(aMatrix))));

   if (handle == 0 || !handle->isValid())
   {
      mexErrMsgIdAndTxt("Damkjer:matAsObj:invalidHandle",
                        "Handle not valid.");
   }

   return handle->object();
}

//*****************************************************************************
// ClassHandle::destroyHandle(const mxArray*)
//>   Free resources for the handle object represented by the MathWorks index.
//
//    @note
//    The mxArray argument is intentionally a non-const pointer to a const
//    mxArray. This signature matches the expected signature for handles passed
//    through the MEX gateway function (mexFunction) on the right-hand side. It
//    is very important that the right-hand side arrays be treated as const and
//    that the memory not be explicitly cleared here. MATLAB will destruct and
//    deallocate the mxArrays when the MEX file is cleared.
//
//    @tparam ObjectT The wrapped object type.
//    @param  aMatrix A MathWorks matrix containing a single index value to be
//                    reinterpretted as a reference to an object of type
//                    ObjectT.
//<
//*****************************************************************************
template<typename ObjectT>
void
ClassHandle<ObjectT>::destroyHandle(const mxArray* aMatrix)
{
   if (mxGetNumberOfElements(aMatrix) != 1    ||
       mxGetClassID(aMatrix) != mxINDEX_CLASS ||
       mxIsComplex(aMatrix))
   {
      mexErrMsgIdAndTxt("Damkjer:destroyHandleTo:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }

   HandleT* handle =
      reinterpret_cast<HandleT*>(*(static_cast<mwIndex*>(mxGetData(aMatrix))));

   if (handle == 0 || !handle->isValid())
   {
      mexErrMsgIdAndTxt("Damkjer:destroyHandleTo:invalidHandle",
                        "Handle not valid.");
   }

   delete handle;
   handle = 0;
}

}
