//*****************************************************************************
// FILE:        SplayTreeAPI.cpp
//
//    Copyright (C)  2012 Kristian Damkjer.
//
// DESCRIPTION:
//>   The application interface for working with VP Trees in MATLAB.
//<
//
// LIMITATIONS:
//>   All interfaces to the VP Tree objects from MATLAB must be routed through
//    this gateway file. This interface is required until and unless MATLAB
//    provides a mechanism for modifying locks on MEX files other than the
//    current file.
//<
//
// SOFTWARE HISTORY:
//> 2013-DEC-13  K. Damkjer
//               Initial Coding.
//<
//*****************************************************************************

#if _OPENMP
#include <omp.h>
#endif

#include "Util/MATLAB/ClassHandle.h"
#include "Util/Dictionaries/SplayTree.h"                  // USES 

#include <vector>
#include <deque>
#include <utility>
#include <sstream>

//***
// Fix "wide char" definition for older versions of MATLAB. This must be placed
// after other includes and before the mex.h include.
//***
#if (defined(MATLAB_MAJOR) && defined(MATLAB_MINOR))
   #if MATLAB_MAJOR <= 7 && MATLAB_MINOR <= 10 && defined(_CHAR16T)
      #define CHAR16_T
   #endif
#endif

#include "mex.h"

//*****************************************************************************
// NAMESPACE:
//>   Splay Tree MEX API types, interfaces, and implementations.
//<
//*****************************************************************************
namespace
{

typedef double WeightT;
   //> The MATLAB SplayTree weight (key) type.

typedef std::deque<WeightT> WeightSetT;
   //> The MATLAB SplayTree weight set type.
   //<

typedef std::pair<WeightT, mwSize> WeightedPointT;
   //> The MATLAB SplayTree weighted point type.
   //<

typedef std::deque<WeightedPointT> WeightedPointSetT;
   //> The MATLAB SplayTree weighted point set type.
   //<

typedef Damkjer::SplayTree<WeightedPointT> TreeT;
   //> The MATLAB SplayTree type.
   //<

typedef Damkjer::ClassHandle<TreeT> HandleT;
   //> The SplayTree MATLAB Class Handle type.
   //<

//*****************************************************************************
// FUNCTION: splayTreeCreate
//>   Create a SplayTree for use in MATLAB.
//
//    New objects are wrapped by the ClassHandle template to ensure that locks
//    are managed correctly to avoid double frees of memory managed by this
//    class, to ensure objects have a life span separate from this function's
//    scope, and to provide convenience methods for passing the pointer to the
//    object between MATLAB and C++.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
splayTreeCreate(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs < 3 || nrhs > 3)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeCreate:nargin",
                        "The splayTreeCreate function requires a two inputs.");
   }
   
   if (nlhs > 1)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeCreate:nargout",
                        "The splayTreeCreate function requires a single output.");
   }

   //***
   // The two inputs required by splayTreeErase is the collection of indices
   // and the collection of weights.
   //***
   const mxArray* weights=prhs[1];
   const mxArray* indices=prhs[2];
   
   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(indices) ||
       mxGetNumberOfDimensions(indices) != 2 ||
       mxIsComplex(indices) ||
       !mxIsNumeric(indices))
   {
      std::stringstream msg;
      msg << ((mxIsSparse(indices)) ? "" : "not ") << "sparse, "
          << (mxGetNumberOfDimensions(indices)) << "-dimensions, "
          << ((mxIsComplex(indices)) ? "" : "not ") << "complex, "
          << ((mxIsNumeric(indices)) ? "" : "not ") << "numeric, "
          << "Index input to splayTreeInsert must be a full vector "
             "of real-valued data representing point indices.";
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        msg.str().c_str());
   }

   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(weights) ||
       mxGetNumberOfDimensions(weights) != 2 ||
       mxIsComplex(weights) ||
       !mxIsNumeric(weights))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        "Weight input to splayTreeInsert must be a full vector "
                        "of real-valued data representing point significance.");
   }

   // Attempt to make the Splay Tree.
   const mwSize iElems = mxGetN(indices);
   const mwSize wElems = mxGetN(weights);

   // Check to make sure that weights are real-valued numerics
   if (iElems != wElems)
   {
      std::stringstream msg;
      msg << "Index elements : " << iElems << "\n"
          << "Weight elements: " << wElems << "\n" 
          << "Must provide exactly one weight for each index.";
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        msg.str().c_str());
   }

   mwIndex* iData = (mwIndex*)mxGetData(indices);
   double*  wData = mxGetPr(weights);

   TreeT* returnTree = new TreeT();

   for (mwSize elem = iElems; elem --> 0;)
   {
      returnTree->insert((WeightedPointT)std::make_pair(wData[elem],
                                                        iData[elem]));
   }
   
   //***
   // Use the ClassHandle named constructor to provide a handle to the new
   // Splay Tree object.
   //***
   try
   {
      plhs[0] = HandleT::createHandle(returnTree);
   }
   catch (const std::exception& e)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeCreate:initError", e.what());
   }
}

//*****************************************************************************
// FUNCTION: splayTreeDestroy
//>   Destroy the SplayTree referenced by a ClassHandle object.
//
//    Destroying objects through the ClassHandle wrapper ensures memory leaks
//    are not introduced through this MEX function and allows the MEX library
//    to be unlocked once all objects have been destroyed and their memory
//    returned to the system.
//
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
inline
void
splayTreeDestroy(int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 2 || !mxIsNumeric(prhs[1]))
   {
       mexErrMsgIdAndTxt("Damkjer:splayTreeDestroy:varargin",
                         "The splayTreeDestroy function requires a single "
                         "input.");
   }

   // Free the Splay Tree through the ClassHandle named destructor.
   HandleT::destroyHandle(prhs[1]);
}

//*****************************************************************************
// FUNCTION: splayTreeHead
//>   Find minumum element in splay tree.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
splayTreeHead(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 2 || !mxIsNumeric(prhs[1]))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeHead:nargin",
                        "The splayTreeHead function requires a single input.");
   }

   if (nlhs != 2)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeHead:nargout",
                        "The splayTreeHead function requires two outputs.");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   TreeT& tree = HandleT::handleReference(prhs[1]);

   const WeightedPointT& head=tree.head();

   plhs[0]=mxCreateNumericMatrix(1, 1, mxDOUBLE_CLASS, mxREAL);
   double* weight = (double*) mxGetData(plhs[0]);
   weight[0] = head.first;

   plhs[1]=mxCreateNumericMatrix(1, 1, mxINDEX_CLASS, mxREAL);
   mwSize* index = (mwSize*) mxGetData(plhs[1]);
   index[0] = head.second;
}

//*****************************************************************************
// FUNCTION: splayTreePopHead
//>   Find minumum element in splay tree.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
splayTreePopHead(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 2 || !mxIsNumeric(prhs[1]))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreePopHead:nargin",
                        "The splayTreePopHead function requires a single input.");
   }

   if (nlhs != 2)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreePopHead:nargout",
                        "The splayTreePopHead function requires two outputs.");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   TreeT& tree = HandleT::handleReference(prhs[1]);

   const WeightedPointT& head=tree.head();

   plhs[0]=mxCreateNumericMatrix(1, 1, mxDOUBLE_CLASS, mxREAL);
   double* weight = (double*) mxGetData(plhs[0]);
   weight[0] = head.first;

   plhs[1]=mxCreateNumericMatrix(1, 1, mxINDEX_CLASS, mxREAL);
   std::size_t* index = (std::size_t*) mxGetData(plhs[1]);
   index[0] = head.second;

   tree.erase(head);
   
   if (tree.find(head) != 0)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreePopHead:nargout",
                        "Pop failed.");
   }
}

//*****************************************************************************
// FUNCTION: splayTreeInsert
//>   Find minumum element in splay tree.
//
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
splayTreeInsert(int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 4 || !mxIsNumeric(prhs[1]))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:nargin",
                        "The splayTreeInsert function requires three inputs.");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   TreeT& tree = HandleT::handleReference(prhs[1]);
  
   //***
   // The two inputs required by splayTreeErase is the collection of indices
   // and the collection of weights.
   //***
   const mxArray* weights=prhs[2];
   const mxArray* indices=prhs[3];
   
   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(indices) ||
       mxGetNumberOfDimensions(indices) != 2 ||
       mxIsComplex(indices) ||
       !mxIsNumeric(indices))
   {
      std::stringstream msg;
      msg << ((mxIsSparse(indices)) ? "" : "not ") << "sparse "
          << (mxGetNumberOfDimensions(indices)) << " "
          << ((mxIsComplex(indices)) ? "" : "not ") << "complex "
          << ((mxIsNumeric(indices)) ? "" : "not ") << "numeric "
          << "Index input to splayTreeInsert must be a full vector "
             "of real-valued data representing point indices.";
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        msg.str().c_str());
   }

   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(weights) ||
       mxGetNumberOfDimensions(weights) != 2 ||
       mxIsComplex(weights) ||
       !mxIsNumeric(weights))
   {
      std::stringstream msg;
      msg << ((mxIsSparse(weights)) ? "" : "not ") << "sparse "
          << (mxGetNumberOfDimensions(weights)) << " "
          << ((mxIsComplex(weights)) ? "" : "not ") << "complex "
          << ((mxIsNumeric(weights)) ? "" : "not ") << "numeric "
          << "Weight input to splayTreeInsert must be a full vector "
                        "of real-valued data representing point significance.";
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        msg.str().c_str());
   }

   // Attempt to make the Splay Tree.
   const mwSize iElems = mxGetN(indices);
   const mwSize wElems = mxGetN(weights);

   // Check to make sure that weights are real-valued numerics
   if (iElems != wElems)
   {
      std::stringstream msg;
      msg << "Index elements : " << mxGetM(indices) << "x" << iElems << "\n"
          << "Weight elements: " << mxGetM(weights) << "x" << wElems << "\n" 
          << "Must provide exactly one weight for each index.";
      mexErrMsgIdAndTxt("Damkjer:splayTreeInsert:prhs",
                        msg.str().c_str());
   }

   mwIndex* iData = (mwIndex*)mxGetData(indices);
   double*  wData = mxGetPr(weights);

   //***
   // Selection of data structure for the point database is not arbitrary.
   //
   // 1. Point coordinates are passed in as double-precision values. Preserve
   //    the data fidelity.
   // 2. Individual points are often low-dimensional. Keep the coordinates for
   //    a single point in contiguous memory by using a std::vector container.
   // 3. The point database can be quite large. Since VP Tree is an online data
   //    structure, allow references to points to occupy non-contiguous memory.
   //    This also speeds construction and destruction of the point database.
   //***
//   WeightedPointSetT pointData(elems);
   
   for (mwSize elem = iElems; elem --> 0;)
   {
      tree.insert((WeightedPointT)std::make_pair(wData[elem], iData[elem]));
   }
}

//*****************************************************************************
// FUNCTION: splayTreeErase
//>   Find minumum element in splay tree.
//
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
splayTreeErase(int nrhs, const mxArray** prhs)
{
   //***
   // Check parameters. Remember that we always have one "extra" input
   // parameter to handle function dispatching through the MEX gateway. It
   // always occupies the first input parameter position.
   //***
   if (nrhs != 4 || !mxIsNumeric(prhs[1]))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeErase:nargin",
                        "The splayTreeErase function requires three inputs.");
   }

   // Retrieve the tree object through the ClassHandle helper method.
   TreeT& tree = HandleT::handleReference(prhs[1]);
  
   //***
   // The two inputs required by splayTreeErase is the collection of indices
   // and the collection of weights.
   //***
   const mxArray* weights=prhs[2];
   const mxArray* indices=prhs[3];
   
   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(indices) ||
       mxGetNumberOfDimensions(indices) != 2 ||
       mxIsComplex(indices) ||
       !mxIsNumeric(indices))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeErase:prhs",
                        "Index input to splayTreeErase must be a full vector "
                        "of real-valued data representing point significance.");
   }

   // Check to make sure that weights are real-valued numerics
   if (mxIsSparse(weights) ||
       mxGetNumberOfDimensions(weights) != 2 ||
       mxIsComplex(weights) ||
       !mxIsNumeric(weights))
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeErase:prhs",
                        "Weight input to splayTreeErase must be a full vector "
                        "of real-valued data representing point significance.");
   }

   // Attempt to make the Splay Tree.
   const mwSize iElems = mxGetN(indices);
   const mwSize wElems = mxGetN(weights);

   // Check to make sure that weights are real-valued numerics
   if (iElems != wElems)
   {
      mexErrMsgIdAndTxt("Damkjer:splayTreeErase:prhs",
                        "Must provide exactly one weight for each index.");
   }

   mwIndex* iData = (mwIndex*)mxGetData(indices);
   double*  wData = mxGetPr(weights);

   //***
   // Selection of data structure for the point database is not arbitrary.
   //
   // 1. Point coordinates are passed in as double-precision values. Preserve
   //    the data fidelity.
   // 2. Individual points are often low-dimensional. Keep the coordinates for
   //    a single point in contiguous memory by using a std::vector container.
   // 3. The point database can be quite large. Since VP Tree is an online data
   //    structure, allow references to points to occupy non-contiguous memory.
   //    This also speeds construction and destruction of the point database.
   //***
//   WeightedPointSetT pointData(elems);
   
   for (mwSize elem = iElems; elem --> 0;)
   {
      tree.erase((WeightedPointT)std::make_pair(wData[elem], iData[elem]));
   }
}

}

//*****************************************************************************
// FUNCTION: mexFunction
//>   The MATLAB Executable Gateway Function.
//
//    Provides the application interface for working with VP Trees in MATLAB.
//    All functional interfaces are routed through this single function to
//    ensure locks and memory management are handled properly by MATLAB.
//
//    @param nlhs the number of left-hand side parameters.
//    @param plhs the array of left-hand side parameters.
//    @param nrhs the number of right-hand side parameters.
//    @param prhs the array of right-hand side parameters.
//<
//*****************************************************************************
void
mexFunction(int nlhs, mxArray** plhs, int nrhs, const mxArray** prhs)
{
   //***
   // This MEX function should always have at least one input parameter to
   // handle function dispatching through the MEX gateway.
   //***
   if (nrhs < 1)
   {
      mexErrMsgIdAndTxt("Damkjer:SplayTreeAPI:nargin",
                        "SplayTreeAPI requires at least one input which "
                        "specifies the operation to be performed.");
   }

   // The operation switch is simply a string.
   char* operation = mxArrayToString(prhs[0]);
   
   if (!operation)
   {
      mexErrMsgIdAndTxt("Damkjer:SplayTreeAPI:InvalidOperation",
                        "Invalid mode supplied to SplayTreeAPI.");
      return; // error should force return.
   }
   
   // Dispatch to helper functions. Err if operation is not recognized.
   if (!strcmp("create", operation))
   {
      splayTreeCreate(nlhs, plhs, nrhs, prhs);
   }
   else if (!strcmp("destroy", operation))
   {
      if (nlhs != 0)
      {
      mexErrMsgIdAndTxt("Damkjer:SplayTreeAPI:nargout",
                        "Nothing to return.");
      }

      splayTreeDestroy(nrhs, prhs);
   }
   else if (!strcmp("head", operation))
   {
      splayTreeHead(nlhs, plhs, nrhs, prhs);
   }
   else if (!strcmp("pop_head", operation))
   {
      splayTreePopHead(nlhs, plhs, nrhs, prhs);
   }
   else if (!strcmp("insert", operation))
   {
      splayTreeInsert(nrhs, prhs);
   }
   else if (!strcmp("erase", operation))
   {
      splayTreeErase(nrhs, prhs);
   }
   else
   {
       mexErrMsgIdAndTxt("Damkjer:SplayTreeAPI:UnknownOperation",
                         "Unrecognized mode provided to SplayTreeAPI.");
   }

   // Prevent a slow memory leak.
   mxFree(operation);
}
