% makeVpTree   Build VpTree spatial indexing library MEX functions and classes.
%
% File:
%    makeVpTree.m
%
% Description:
%    This MATLAB function provides an simplified interface for building the
%    VpTree C++ and MEX files.
%
% Limitations:
%    Assumes mex -setup has been successfully run.
%
% Synopsis:
%    makeVpTree(options...)
%
% Inputs:
%   Option strings may be provided as abbreviations as long as they resolve to
%   a unique option.
%
%   'debug'     - Compile with debug flags and fully populated symbol table.
%   'parallel'  - Enable OpenMP pragmas for multi-threaded computing.
%   'verbose'   - Provide verbose output while compiling.
%   'warnings'  - Provide compiler warning output for production-ready code.
%                 Implies 'verbose'.
%
% Outputs:
%    None.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    VpTree
%

% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

function makeVpTree(varargin)

command = 'mex -largeArrayDims -DTEMPLATE_INCLUSION';

matlab_ver = regexp(version, '[. ]', 'split');
command = [command...
           ' -DMATLAB_MAJOR=' matlab_ver{1}...
           ' -DMATLAB_MINOR=' matlab_ver{2}...
           ' -DMATLAB_REVIS=' matlab_ver{3}...
           ' -DMATLAB_BUILD=' matlab_ver{4}...
           ' -DMATLAB_REL=' matlab_ver{5}(2:end-1)];

flags = {'verbose', 'debug', 'warnings', 'parallel'};
options = '';

isVerbose = 0;

while ~isempty(varargin)
   flag = varargin{1};

   % If the property has been supplied in a shortened form, lengthen it
   iFlag = find(strncmpi(flag, flags, length(flag)));
   
   if isempty(iFlag)
      % Pass through any arguments we don't recognize
      options = strcat(options, [' ' flag]);
      
      % Alternatively, escalate to an error.
      %error('Damkjer:makeVpTree:InvalidFlag', ['Invalid Flag: ' flag]);
   elseif length(iFlag) > 1
      error('Damkjer:makeVpTree:AmbiguousFlag', ...
            ['Supplied shortened flag is ambiguous: ' flag]);
   else
      flag = flags{iFlag};
      
      switch flag
      case 'verbose'
         command = strcat(command, ' -v');
         isVerbose = 1;
      case 'debug'
         command = strcat(command, ' -g');
      case 'parallel'
         if (ispc)
            command = strcat(command, ' COMPFLAGS="$COMPFLAGS /openmp"');
         else
            command = strcat(command, ' CFLAGS="\$CFLAGS -fopenmp"',...
                                      ' CXXFLAGS="\$CXXFLAGS -fopenmp"',...
                                      ' LDFLAGS="\$LDFLAGS -fopenmp"');
         end
      case 'warnings'
         if (ispc)
            command = strcat(command, ' -v COMPFLAGS="$COMPFLAGS -W3"');
         else
            command = strcat(command, ' -v',...
                                      ' CFLAGS="\$CFLAGS -Wall -Wextra"',...
                                      ' CXXFLAGS="\$CXXFLAGS -Wall -Wextra"');
         end
      otherwise
         error('Damkjer:makeVpTree:NotImplementedFlag', ...
               ['Flag recognized, but not supported: ' flag]);
      end
   end
   
   varargin(1) = [];
end

[path,~,~]=fileparts(mfilename('fullpath'));
command = [command options ' -I' fullfile(path,'..','..','..')...
           ' -outdir ' path ' ' path '/'];

disp([command 'VpTreeAPI.cpp']);

eval([command 'VpTreeAPI.cpp']);

end
